Rem
Rem $Header: ess/sql/common_oracle/sqljob/ess_plsql_runtime.sql /st_pcbpel_12.2.1.4.0soabp/2 2023/02/22 14:35:51 apfwkr Exp $
Rem
Rem ess_plsql_runtime.sql
Rem
Rem Copyright (c) 2006, 2023, Oracle and/or its affiliates. 
Rem All rights reserved.
Rem
Rem  NAME
Rem    ess_plsql_runtime.sql - ESS runtime schema setup using PLSQL logic
Rem
Rem  DESCRIPTION
Rem    This script creates database objects pertaining to the ESS PL/SQL
Rem    API and jobs feature when the ESS runtime schema is created. Tasks that
Rem    pertain to ESS runtime schema in general are done by ess_runtime.sql.
Rem    Setup for AQ/Scheduler notification done by ess_scheduler_notify.sql.
Rem
Rem  NOTES
Rem    This script should be run in the context of the ESS schema.
Rem    It assumes that either ALTER SESSION SET CURRENT_SCHEMA has been
Rem    done or it is being run as the ESS schema owner.
Rem
Rem    See ess/notes/ess_sql_error.txt for information about SQL errors
Rem    raised by this script. That file should be updated when messages
Rem    are added, changed, and so on.
Rem
Rem  MODIFIED   (MM/DD/YY)
Rem   pallacha   09/27/21 - Backport pallacha_bug-33351665_fa from main
Rem   pallacha   09/14/21 - fix bug 33351665
Rem   apfwkr     07/17/20 - Backport pallacha_bug-31443286_12215 from main
Rem   pallacha   07/08/20 - Add fix for ATP-DB
Rem   chall      03/25/16 - move aqagent_name init to ess_scheduler_notify.sql;
Rem                         remove use of temporary procedures
Rem   chall      02/19/15 - Scheduler job class grant done as rcu admin for
Rem                         Database Vault
Rem   chall      09/24/14 - revise comments
Rem   chall      07/16/10 - remove support for AQ-based work notification
Rem   chall      12/01/09 - load package specs, bodies, etc in ess_runtime.sql
Rem   kbittler   10/02/09 - move transformation functions to new file
Rem   kbittler   10/02/09 - no longer target finalization to original processor
Rem   chall      04/08/09 - ess_version package now in essversion.sql
Rem   chall      03/20/09 - call ess_view.sql, psiversion.sql scripts
Rem   chall      03/13/09 - essi_fusion package support
Rem   chall      11/16/08 - support for plsql runas
Rem   chall      10/28/08 - support SQL Scheduler job owner is non-ESS schema
Rem                         user; remove DBMS_XS_SESSIONS support
Rem   ddavison   07/10/08 - unable to set correlation id
Rem   kbittler   05/30/08 - multi-instance supprot
Rem   chall      05/07/08 - rename paused_flag, resumed_flag fields
Rem   kbittler   12/18/07 - 
Rem   chall      12/18/07 - enhance transformation for paused, warning
Rem   chall      12/14/07 - add groupName property to REQUEST_FINIALIZATION
Rem                         event
Rem   kbittler   12/11/07 - add PL/SQL runtime API package
Rem   ddavison   12/06/07 - bug 6641064: event worker looping
Rem   chall      08/29/07 - bugfix 6365435; remove standalone helper routines
Rem   chall      07/24/07 - bugfix 6000335; multiple schema support
Rem
Rem ========================================================================


-- ========================================================================
-- Establish ESS prefix for this schema.
--
-- Tasks:
-- a) Establish ESS prefix used for Scheduler objects associated with this
--    ESS schema. This is done by creating an Oracle Scheduler job class
--    that will be the default job class associated with this ESS schema.
--    ESS prefix value will be stored in ess_config.
--
-- NOTE: Starting 12.2.1.1.0, establishing AQ agent name is deferred to
-- ess_scheduler_notify.sql script since we only need an AQ agent if
-- AQ notification callback is used to receive Oracle Scheduler event
-- messages. Trying to move away from that model for several reasons;
-- extra complexity needed for the Scheduler/AQ setup tasks, especially
-- the AQ notification registration, extra privileges needed to do that
-- setup, notification issues due to various AQ bugs, and so on.
--
-- Background Information:
--
-- We determine a "unique" prefix to use for Oracle Scheduler object names
-- associated with this ESS schema; for example, Scheduler job name and
-- job class name.
--
-- The Oracle Scheduler job for an ESS request will likely be owned by
-- someone other than an ESS schema user leaving the ESS prefix used for
-- the Scheduler job name as the last meaningful thing to use for the AQ
-- subscriber rule to propagate Scheduler event messages for an ESS schema.
-- Note: AQ subscriber needed only if AQ notification callback is used.
--
-- A GRANT EXECUTE on the Scheduler job class to PUBLIC must be done so
-- the job class can be used by any Oracle Scheduler job created by ESS.
-- Oracle Scheduler job owner must have execute privilege on the job class
-- used for a Scheduler job. Scheduler job class objects are owned by SYS.
-- The grant will be deferred to later step so can be done as RCU admin.
-- An "ORA-47401: Realm violation ...." would occur if grant is done as
-- ESS schema user for a Database Vault enabled db due to being SYS object.
--
-- Naming convention for Oracle Scheduler/AQ objects associated with
-- this ESS schema:
--
-- * Default Scheduler job class
--   Format: <ess_prefix>$JC_DEFAULT
--   Example:  ESS1$JC_DEFAULT
--
-- * Default Scheduler job name
--   Format: <ess_prefix>$JB_<ess_request_id>
--   Example:  ESS1$JB_12345
--
-- * AQ Agent Name for ESS Schema
--   Format:  <ess_schema>_$ESS
--   Example: DEV_ESS_$ESS
--
-- ========================================================================

declare
  v_debug boolean := true;
  v_ess_schema varchar2(30) := sys_context('userenv', 'current_schema');
  v_ess_prefix varchar2(30) := null;

  -- Looks for an unused prefix that ties Oracle Scheduler object names to
  -- an ESS schema. Returns null if unable to find an unused prefix.
  function get_unused_ess_prefix
  return varchar2
  as
    vi_debug boolean := false;
    vi_method varchar2(30) := 'get_unused_ess_prefix';
    vi_max_ess_schemas pls_integer := 200;  -- max number of ESS schemas per db
    vi_free_prefix boolean := false;        -- t=found free/available prefix
    vi_ess_prefix  varchar2(30) := null;
    vi_name varchar2(30) := null;
    vi_idx pls_integer;
    vi_null char(1);
  begin
    -- Look for an available 'ESS<nnn>$JC_DEFAULT' Scheduler job class name.
    for vi_idx in 1..vi_max_ess_schemas loop
      vi_ess_prefix := 'ESS'||vi_idx;
      if vi_debug then
        dbms_output.put_line(vi_method||': checking prefix: '||vi_ess_prefix);
      end if;

      -- ESS default job class name
      vi_name := vi_ess_prefix||'$JC_DEFAULT';
      begin
        select null into vi_null from all_scheduler_job_classes jc
          where job_class_name = vi_name;
        if vi_debug then
          dbms_output.put_line(vi_method||': job class already exists: '||vi_name);
        end if;
      exception
        when NO_DATA_FOUND then
          vi_free_prefix := true;
          exit;
      end;
    end loop;

    if not vi_free_prefix then
      vi_ess_prefix := null;
    end if;

    if vi_debug then
      dbms_output.put_line(vi_method||': returning prefix: '||vi_ess_prefix);
    end if;
    return vi_ess_prefix;
  end get_unused_ess_prefix;

  -- Create default Scheduler job class associated with the ESS schema.
  -- Note: The GRANT EXECUTE for Scheduler job class object will be
  -- done later as RCU admin user. See comments at top of file.
  procedure create_sched_job_class
  ( p_ess_schema in varchar2,
    p_ess_prefix in varchar2,
    p_job_class out varchar2 )
  as
    vi_method varchar2(30) := 'create_sched_job_class';
    vi_debug boolean := true;
    vi_job_class varchar2(30) := null;
    vi_comment varchar2(500) := null;
    vi_stmt varchar2(300) := null;

    e_SCHED_ALREADY_EXISTS exception;
    pragma EXCEPTION_INIT(e_SCHED_ALREADY_EXISTS, -27477);
  begin
    p_job_class := null;
    begin
      -- create Oracle Scheduler job class
      vi_job_class := p_ess_prefix||'$JC_DEFAULT';
      vi_comment := 'Default ESS job class for '||p_ess_schema||' schema';
      dbms_scheduler.create_job_class(job_class_name => vi_job_class,
                                      comments => vi_comment);
      p_job_class := vi_job_class;
    exception
      when e_SCHED_ALREADY_EXISTS then
        if vi_debug then
          dbms_output.put_line(
              vi_method||': ESS prefix is already in use: '||p_ess_prefix);
          dbms_output.put_line(SQLERRM);
        end if;
    end;
  end create_sched_job_class;

  -- Establish sqljob prefix for the ESS schema.
  -- a) Determine ESS prefix to associate with the ESS schema.
  -- b) Create default Oracle Scheduler job class associated with ESS schema.
  -- c) Save the ESS sqljob prefix to ess_config table.
  procedure init_sqljob_prefix
  ( p_ess_schema in varchar2,
    p_ess_prefix out varchar2 )
  as
    vi_debug boolean := true;
    vi_method varchar2(30) := 'init_sqljob_prefix';
    vi_max_attempts pls_integer := 3;
    vi_ess_prefix varchar2(30) := null;
    vi_job_class varchar2(30) := null;
    vi_created_job_class boolean := false;
    vi_idx pls_integer;
    v_atp_value varchar2(30) := null;
    v_stmt varchar2(500) := null;
    v_is_atp boolean := false;


  begin
    -- Allow a couple attempts at this in case different ESS schemas are
    -- being created concurrently, they might try to use the same ess prefix.
    p_ess_prefix := null;

    for vi_idx in 1..vi_max_attempts loop
      vi_ess_prefix := get_unused_ess_prefix();
      exit when vi_ess_prefix is null;    -- hmm...all prefixes in use??

      -- create default Scheduler job class associated with this ESS schema
      create_sched_job_class(p_ess_schema, vi_ess_prefix, vi_job_class);

      -- Save the sqljob prefix associated with this ESS schema.
      if vi_job_class is not null then
        vi_created_job_class := true;
        begin
          update ess_config set sqljob_prefix = vi_ess_prefix;
          commit;
          exit;    -- all done, break out of loop
        exception
          when others then
            rollback;
            dbms_scheduler.drop_job_class(job_class_name=>vi_job_class);
            raise;
        end;
      end if;
    end loop;

-- For ATP DV enabled DB, now the realm violation exception is not thrown and jobclass can be granted execute permission to public.
-- NO_DATA_FOUND : Below result gives No data for non-ATP DB, hence the ignoring the below exception.
-- Only for Non ATP DB executing the grant on job class to public in postdataload as existing framework. 
-- ATP DB - grating permission while creating the job class only.

    if vi_created_job_class then
      begin
          dbms_output.put_line('dbms_db_version.version: '|| dbms_db_version.version);
          if (dbms_db_version.version) < '18' then
             dbms_output.put_line('This looks to be a non ATP database and can be ignored.');
             dbms_output.put_line('DB version is less than 18');
          else
             dbms_output.put_line('DB version is great than 18');
             -- select sys_context('userenv','cloud_service') into v_atp_value from dual;
             v_atp_value := nvl(lower(sys_context('userenv','cloud_service')), 'null');
             dbms_output.put_line('v_atp_value: '||v_atp_value);
             if (v_atp_value in ('dwcs','oltp','jdcs','paas')) then
                 v_stmt := 'grant execute on SYS.'||vi_job_class||' to public';
                 dbms_output.put_line('grant execute on jobclass to public: '||v_stmt);
                 execute immediate v_stmt;
             else
                 dbms_output.put_line('This looks to be a non ATP database and can be ignored.');
             end if;
          end if;
       exception
          when NO_DATA_FOUND then
             dbms_output.put_line('This looks to be a non ATP database and can be ignored.');
       end;
    end if;

    if not vi_created_job_class then
      raise_application_error(-20250,
        'unable to establish ESS prefix for '||p_ess_schema||' schema');
    end if;

    if vi_debug then
      dbms_output.put_line(vi_method||': sqljob_prefix='||vi_ess_prefix ||
                           ', default job class='||vi_job_class);
    end if;

    p_ess_prefix := vi_ess_prefix;
  end init_sqljob_prefix;

begin
  select sqljob_prefix into v_ess_prefix from ess_config;

  if v_ess_prefix is not null then
    if v_debug then
      dbms_output.put_line(
          'ESS SQL prefix already set to ' || v_ess_prefix ||
          ' for ' || v_ess_schema || ' schema');
    end if;
  else
    init_sqljob_prefix(v_ess_schema, v_ess_prefix);
    if v_debug then
      dbms_output.put_line('ESS SQL job prefix initialized: ' ||
                           'ess_schema='||v_ess_schema ||
                           ', sqljob_prefix='||v_ess_prefix);
    end if;
  end if;
end;
/

Rem === End file: ess_plsql_runtime.sql ====================================
