/*
*
* deleteComponentInstances.sql
*
* Copyright (c) 2010, 2020, Oracle and/or its affiliates. 
*
*    NAME
*      deleteComponentInstances.sql - BPEL purge script 
*
*
*    DESCRIPTION
*     This script is used to purge BPEL tables provided with a list of composite instance ids
*     in an idTable. 
*
*    For a user with multi cpu machine, usage of parallel dml may help. Add this to master script.
*      execute immediate 'alter session enable parallel dml';
*
*/

PROCEDURE truncate_temp_tables
AS
BEGIN
  --before starting clean up temp tables
  EXECUTE IMMEDIATE 'TRUNCATE TABLE temp_cube_instance';
  EXECUTE IMMEDIATE 'TRUNCATE TABLE temp_document_ci_ref';
  EXECUTE IMMEDIATE 'TRUNCATE TABLE temp_document_dlv_msg_ref';
  EXECUTE IMMEDIATE 'TRUNCATE TABLE temp_xml_document';
END truncate_temp_tables;


/*
* Procedure: purge_bpel_tables
*
* Description
* This procedure purges following BPEL tables based on temp tables created in earlier step.
*
*    HEADERS_PROPERTIES
*    AG_INSTANCE
*    TEST_DETAILS
*    CUBE_SCOPE
*    AUDIT_COUNTER
*    AUDIT_TRAIL
*    AUDIT_DETAILS
*    CI_INDEXS
*    WORK_ITEM
*    WI_FAULT
*    XML_DOCUMENT
*    DOCUMENT_DLV_MSG_REF
*    DOCUMENT_CI_REF
*    DLV_MESSAGE
*    DLV_SUBSCRIPTION
*    CUBE_INSTANCE
*    
*    BPM_AUDIT_QUERY
*    BPM_MEASUREMENT_ACTIONS
*    BPM_MEASUREMENT_ACTION_EXCEPS
*    BPM_CUBE_AUDITINSTANCE
*    BPM_CUBE_TASKPERFORMANCE
*    BPM_CUBE_PROCESSPERFORMANCE
*
*/
PROCEDURE purge_bpel_tables(purge_partitioned_component in boolean default false, ignore_state in boolean default false)
AS
  v_stmt  VARCHAR2(2000);
  batch_total_count integer := 0;
BEGIN
--Create additional temp tables
v_stmt := 'Create temp_document_ci_ref' ;
    INSERT INTO temp_document_ci_ref (document_id) SELECT document_id
        FROM document_ci_ref d, temp_cube_instance t WHERE d.cikey = t.cikey;
    debug_purge('temp_document_ci_ref', 'Inserted = ');
    COMMIT;

v_stmt := 'Create temp_document_dlv_msg_ref from dlv_message type = 1 and 2';
    /*
    INSERT INTO temp_document_dlv_msg_ref (message_guid,document_id) SELECT m.message_guid,ddmr.document_id
           FROM document_dlv_msg_ref ddmr, dlv_message m , temp_cube_instance t
           WHERE ddmr.message_guid = m.message_guid
                    AND m.state in (2,3) AND ( m.cikey = t.cikey OR m.cikey = 0 ) ;
    COMMIT;
    */
    if ignore_state
    then
        INSERT INTO temp_document_dlv_msg_ref (message_guid,document_id)
            SELECT m.message_guid,ddmr.document_id
               FROM document_dlv_msg_ref ddmr, dlv_message m , temp_cube_instance t
               WHERE ddmr.message_guid = m.message_guid
                        AND m.cikey = t.cikey ;
    else
        INSERT INTO temp_document_dlv_msg_ref (message_guid,document_id)
            SELECT m.message_guid,ddmr.document_id
               FROM document_dlv_msg_ref ddmr, dlv_message m , temp_cube_instance t
               WHERE ddmr.message_guid = m.message_guid
                        AND m.state in (2,3) AND m.cikey = t.cikey ;
    /*Include expired messages as well */
    INSERT INTO temp_document_dlv_msg_ref (message_guid,document_id)
            SELECT m.message_guid,ddmr.document_id
               FROM document_dlv_msg_ref ddmr, dlv_message m , temp_cube_instance t
               WHERE ddmr.message_guid = m.message_guid
                        AND m.expiry_date < SYSDATE AND m.cikey = t.cikey ;
    end if;

    debug_purge('temp_document_dlv_msg_ref', 'Inserted = ');
    COMMIT;
/*
    DELETE FROM temp_document_ci_ref doc WHERE doc.document_id IN (select b2b.document_id from B2B_DATA_STORAGE b2b);
    COMMIT;
    DELETE FROM temp_document_dlv_msg_ref doc WHERE doc.document_id IN (select b2b.document_id from B2B_DATA_STORAGE b2b);
    COMMIT;
    DELETE FROM TEMP_XML_DOCUMENT_WSDL_PACK doc WHERE doc.document_id IN (select b2b.document_id from B2B_DATA_STORAGE b2b);
    COMMIT;
*/
-- Being purge of BPEL data

v_stmt := 'Purge HEADERS_PROPERTIES' ;
     batch_total_count := batch_delete_func('DELETE FROM HEADERS_PROPERTIES WHERE message_guid IN (SELECT message_guid FROM temp_document_dlv_msg_ref)');
     debug_purge2('HEADERS_PROPERTIES',batch_total_count);
     COMMIT;

v_stmt := 'Purge AG_INSTANCE' ;
    batch_total_count := batch_delete_func('DELETE FROM AG_INSTANCE WHERE cikey in (SELECT tpic.cikey FROM temp_cube_instance tpic)');
    debug_purge2('AG_INSTANCE',batch_total_count);
    COMMIT;

v_stmt := 'Purge TEST_DETAILS' ;
    batch_total_count := batch_delete_func('DELETE FROM TEST_DETAILS WHERE cikey in (SELECT tpic.cikey FROM temp_cube_instance tpic)');
    debug_purge2('TEST_DETAILS',batch_total_count);
    COMMIT;
    
v_stmt := 'Purge CUBE_SCOPE' ;
   if purge_partitioned_component = true OR is_table_partitioned('CUBE_SCOPE','CI_PARTITION_DATE') = FALSE then
		batch_total_count := batch_delete_func('DELETE FROM CUBE_SCOPE WHERE cikey in (SELECT tpic.cikey FROM temp_cube_instance tpic)');
		debug_purge2('CUBE_SCOPE',batch_total_count);
		COMMIT;
	end if;	

v_stmt := 'Purge AUDIT_COUNTER' ;
   batch_total_count := batch_delete_func(' DELETE FROM AUDIT_COUNTER WHERE cikey in (SELECT tpic.cikey FROM temp_cube_instance tpic)');
    debug_purge2('AUDIT_COUNTER',batch_total_count);
    COMMIT;

v_stmt := 'Purge AUDIT_TRAIL' ;
    batch_total_count := batch_delete_func('DELETE FROM AUDIT_TRAIL WHERE cikey in (SELECT tpic.cikey FROM temp_cube_instance tpic)');
    debug_purge2('AUDIT_TRAIL',batch_total_count);
    COMMIT;

v_stmt := 'Purge AUDIT_DETAILS' ;
    batch_total_count := batch_delete_func('DELETE FROM AUDIT_DETAILS WHERE cikey in (SELECT tpic.cikey FROM temp_cube_instance tpic)');
    debug_purge2('AUDIT_DETAILS',batch_total_count);
    COMMIT;

v_stmt := 'Purge CI_INDEXS' ;
    batch_total_count := batch_delete_func('DELETE FROM CI_INDEXES WHERE cikey in (SELECT tpic.cikey FROM temp_cube_instance tpic)');
    debug_purge2('CI_INDEXES',batch_total_count);
    COMMIT;

v_stmt := 'Purge WORK_ITEM' ;
    batch_total_count := batch_delete_func('DELETE FROM WORK_ITEM WHERE cikey in (SELECT tpic.cikey FROM temp_cube_instance tpic)');
    debug_purge2('WORK_ITEM',batch_total_count);
    COMMIT ;

v_stmt := 'Purge WI_FAULT' ;
    batch_total_count := batch_delete_func('DELETE FROM WI_FAULT WHERE cikey in (SELECT tpic.cikey FROM temp_cube_instance tpic)');
    debug_purge2('WI_FAULT',batch_total_count);
    COMMIT ;

v_stmt := 'Purge XML_DOCUMENT';
    -- TODO optimize the if condition . Should not call is_table_partitioned for each loop)  
    if purge_partitioned_component = true OR is_table_partitioned('XML_DOCUMENT','DOC_PARTITION_DATE') = FALSE then
    batch_total_count := batch_delete_func('DELETE FROM  XML_DOCUMENT d WHERE document_id in
              ( SELECT t1.document_id FROM  temp_document_dlv_msg_ref t1
                union  
                SELECT t2.document_id FROM temp_document_ci_ref t2
              )');
    debug_purge2('XML_DOCUMENT',batch_total_count);
    COMMIT ;
	end if;
    
v_stmt := 'Purge DOCUMENT_DLV_MSG_REF' ;
    batch_total_count := batch_delete_func('DELETE FROM DOCUMENT_DLV_MSG_REF  WHERE document_id in (SELECT document_id FROM temp_document_dlv_msg_ref t)');
    debug_purge2('DOCUMENT_DLV_MSG_REF',batch_total_count);
    COMMIT ;

v_stmt := 'Purge DOCUMENT_CI_REF' ;
    batch_total_count := batch_delete_func('DELETE FROM DOCUMENT_CI_REF WHERE cikey in (SELECT tpic.cikey FROM temp_cube_instance tpic)');
    debug_purge2('DOCUMENT_CI_REF',batch_total_count);
    COMMIT ;

v_stmt := 'Purge DLV_MESSAGE' ;
    batch_total_count := batch_delete_func('DELETE FROM DLV_MESSAGE  where message_guid in ( SELECT message_guid from temp_document_dlv_msg_ref )');
    debug_purge2('DLV_MESSAGE',batch_total_count);
    COMMIT ;

v_stmt := 'Purge DLV_SUBSCRIPTION';
    batch_total_count := batch_delete_func('DELETE FROM DLV_SUBSCRIPTION s WHERE cikey in (SELECT tpic.cikey FROM temp_cube_instance tpic)');
    debug_purge2('DLV_SUBSCRIPTION',batch_total_count);
    COMMIT ;

v_stmt := 'Purge DLV_AGGREGATION';
    batch_total_count := batch_delete_func('DELETE FROM DLV_AGGREGATION WHERE cikey in (SELECT tpic.cikey FROM temp_cube_instance tpic)');
    debug_purge2('DLV_AGGREGATION',batch_total_count);
    COMMIT ;

v_stmt := 'Purge CUBE_INSTANCE';
    IF purge_partitioned_component = true OR is_table_partitioned('CUBE_INSTANCE','CPST_INST_CREATED_TIME') = false then
    -- TODO optimize the if condition . Should not call is_table_partitioned for each loop)   
    batch_total_count := batch_delete_func('DELETE FROM cube_instance ci WHERE cikey in (SELECT tpic.cikey FROM temp_cube_instance tpic)');
    debug_purge2('CUBE_INSTANCE',batch_total_count);
    COMMIT ;
    END IF;
    -- ============ BPMN Tables ============

    IF table_exist('BPM_AUDIT_QUERY') THEN
       v_stmt := 'Purge BPM_AUDIT_QUERY' ;
       batch_total_count := batch_delete_func('DELETE FROM BPM_AUDIT_QUERY WHERE component_instance_id in (SELECT tpic.cikey FROM temp_cube_instance tpic)');
       debug_purge2('BPM_AUDIT_QUERY',batch_total_count);
       COMMIT ;
    END IF;

    IF table_exist('BPM_MEASUREMENT_ACTIONS') THEN
       v_stmt := 'Purge BPM_MEASUREMENT_ACTIONS'; 
       batch_total_count := batch_delete_func('DELETE FROM BPM_MEASUREMENT_ACTIONS WHERE component_instance_id in (SELECT tpic.cikey FROM temp_cube_instance tpic)');
       debug_purge2('BPM_MEASUREMENT_ACTIONS',batch_total_count);
       COMMIT ;
    END IF;

    IF table_exist('BPM_MEASUREMENT_ACTION_EXCEPS') THEN
       v_stmt := 'Purge BPM_MEASUREMENT_ACTION_EXCEPS';
       batch_total_count := batch_delete_func('DELETE FROM BPM_MEASUREMENT_ACTION_EXCEPS WHERE component_instance_id in (SELECT tpic.cikey FROM temp_cube_instance tpic)');
       debug_purge2('BPM_MEASUREMENT_ACTION_EXCEPS',batch_total_count);
       COMMIT ;
    END IF;

    IF table_exist('BPM_CUBE_AUDITINSTANCE') THEN
       v_stmt := 'Purge BPM_CUBE_AUDITINSTANCE';
       batch_total_count := batch_delete_func('DELETE FROM BPM_CUBE_AUDITINSTANCE WHERE componentinstanceid in (SELECT tpic.cikey FROM temp_cube_instance tpic)');
       debug_purge2('BPM_CUBE_AUDITINSTANCE',batch_total_count);
       COMMIT ;
    END IF;

    IF table_exist('BPM_CUBE_TASKPERFORMANCE') THEN
       v_stmt := 'Purge BPM_CUBE_TASKPERFORMANCE';
       batch_total_count := batch_delete_func('DELETE FROM BPM_CUBE_TASKPERFORMANCE WHERE componentinstanceid in (SELECT tpic.cikey FROM temp_cube_instance tpic)');
       debug_purge2('BPM_CUBE_TASKPERFORMANCE',batch_total_count);
       COMMIT ;
    END IF;

    IF table_exist('BPM_CUBE_PROCESSPERFORMANCE') THEN
       v_stmt := 'Purge BPM_CUBE_PROCESSPERFORMANCE';
       batch_total_count := batch_delete_func('DELETE FROM BPM_CUBE_PROCESSPERFORMANCE WHERE componentinstanceid in (SELECT tpic.cikey FROM temp_cube_instance tpic)');
       debug_purge2('BPM_CUBE_PROCESSPERFORMANCE',batch_total_count);
       COMMIT ;
    END IF;
    
EXCEPTION
  when others then
    log_error(v_stmt);
    raise;

END purge_bpel_tables;
/*
* Procedure : deleteComponentInstances
*
* Description: 
*  This procedure is provided a table with composite instance ids that are purge able.
*  Following temp tables are created and passed to purge_bpel_tables procedure.
*
*  1. temp_cube_instance
*  2. temp_document_ci_ref
*  3. temp_document_dlv_msg_ref
*
*
*  Parameters:
*  idTable : Name of the id table that contains list of purge able composite instance IDs
*  
*/

PROCEDURE deleteComponentInstances(p_id_table   IN VARCHAR2, purge_partitioned_component in boolean default false, ignore_state in boolean default false)
AS
  v_stmt  VARCHAR2(2000);
BEGIN

  truncate_temp_tables();
v_stmt := 'INSERT INTO temp_cube_instance(CIKEY) ' ||
              'SELECT cikey FROM cube_instance ci, '||p_id_table || ' comp WHERE ci.ecid = comp.ECID';
EXECUTE immediate v_stmt;
debug_purge('temp_cube_instance', 'Inserted = ');
COMMIT;

    purge_bpel_tables(purge_partitioned_component, ignore_state);

EXCEPTION
  when others then
    log_error(v_stmt);
    raise;

END deleteComponentInstances;
