/*
Rem
Rem Copyright (c) 2006, 2024, Oracle and/or its affiliates. 
Rem All rights reserved.
Rem
Rem    NAME
Rem      deleteInstances.sql <one-line expansion of the name>
Rem
Rem    DESCRIPTION
Rem      SOA Purge
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem       apfwkr   02/22/24 - CI# 36329882 of
Rem       apfwkr   02/22/24 - apfwkr_blr_backport_28642751_12.2.1.4.231122soabp
Rem                           from st_pcbpel_12.2.1.4.0soabp
Rem       apfwkr   02/12/24 - BLR# 36283216 of wfchen_bug-28642751_12215 .
Rem       apfwkr   02/12/24 - BLR# 36283216
Rem       apfwkr   02/12/24 - apfwkr_blr_backport_28642751_12.2.1.4.231122soabp
Rem                           from main
Rem       apfwkr   11/19/20 - Backport
Rem                           apfwkr_blr_backport_22329267_12.2.1.4.200917soabp
Rem                           from st_pcbpel_12.2.1.4.0soabp
Rem       apfwkr   11/18/20 - Backport apfwkr_blr_backport_22329267_12.2.1.4.0
Rem                           from main
Rem       apfwkr   05/14/20 - Backport linlxu_bug-29750407_12212-main from main
Rem       apfwkr   04/28/20 - Backport linlxu_bug-30490102_12212-main from main
Rem       apfwkr   11/16/20 - Backport linlxu_bug-22329267_12212-main from main
Rem       bpulappa 02/03/16 - XbranchMerge bpulappa_bug-22451485_r2 from
Rem                           st_pcbpel_pt-next
Rem       sanjain  04/29/10 - Enabling Mediator purge
Rem
*/
    function  getComponentPartitionInfo return component_partition_info is
     componentPartInfo component_partition_info ;
    begin
        log_info('checking for  partitions');
        componentPartInfo := component_partition_info(); 
        componentPartInfo.fabricPartitioned := soa11g_fabric.getComponentPartitionInfo;
        componentPartInfo.bpelPartitioned := soa11g_orabpel.getComponentPartitionInfo;
	componentPartInfo.mediatorPartitioned := soa11g_mediator.getComponentPartitionInfo;
        if soa11g_b2b.isComponentPartitioned then
           componentPartInfo.b2bPartitioned := 'Y';
        else
           componentPartInfo.b2bPartitioned := 'N';
        end if;
        if soa11g_workflow.isComponentPartitioned then
           componentPartInfo.workflowPartitioned := 'Y';
        else
           componentPartInfo.workflowPartitioned := 'N';
        end if;
        if soa11g_decision.isComponentPartitioned then
           componentPartInfo.decisionPartitioned := 'Y';
        else
           componentPartInfo.decisionPartitioned := 'N';
        end if;
        log_info('done checking for partitions');
        return componentPartInfo;
    end;
 
  function make_composite_dn(soa_partition_name in varchar2, composite_name in varchar2, composite_revision in varchar2) return varchar2 is
     composite_dn varchar2(500);
  
    begin
        if composite_name is not null then
            composite_dn := composite_name||'!';
        end if; 
        
        if composite_name is not null and composite_revision is not null then
            composite_dn := composite_dn||composite_revision;
        end if;
        
        if soa_partition_name is not null then
            composite_dn := soa_partition_name || '/' || composite_dn ; 
        end if;
        
        if soa_partition_name is null and composite_dn is not null then
            composite_dn := 'default/'||composite_dn;
        end if;
        
        return composite_dn;    
	    
    end make_composite_dn;
      
    PROCEDURE dangling_delete(retention_period in date default null, p_transaction_size NUMBER, p_stop_time INTEGER, write_file in utl_file.file_type default null, keep_workflow_inst in boolean default true) is 
     p_retention_period date := retention_period;
     v_stoptime date := sysdate + NVL(p_stop_time,24*60)/(24*60);
    begin
       IF (p_retention_period IS NULL) THEN
          p_retention_period := sysdate -7;
       END IF;

      IF (sysdate >= v_stoptime) THEN
           return;
      END IF;

       soa11g_orabpel.dangling_delete(p_retention_period, p_transaction_size, v_stoptime, write_file);
      IF (sysdate >= v_stoptime) THEN
           return;
      END IF;
       soa11g_mediator.dangling_delete(p_retention_period, p_transaction_size, v_stoptime, write_file);
      IF (sysdate >= v_stoptime) THEN
           return;
      END IF;
      if (keep_workflow_inst = false) then   --by default no workflow cleanup
		log_info(' calling soa_workflow.dangling_delete, as keep_workflow_inst is false');
		soa11g_workflow.dangling_delete(p_retention_period, p_transaction_size, v_stoptime, write_file);
      end if;
      IF (sysdate >= v_stoptime) THEN
           return;
      END IF;
       soa11g_fabric.dangling_delete(p_retention_period, p_transaction_size, v_stoptime, write_file);
      IF (sysdate >= v_stoptime) THEN
           return;
      END IF;
    end dangling_delete;

 procedure pruneOpenECIDs (purge_id_table in varchar2,prune_running_insts_table in varchar2, retention_period in timestamp, write_file in utl_file.file_type default null) is 

     v_stmt  VARCHAR2(500);

     begin

       -- Top Level pruning of ECID that might have been migrated to 12c
       -- and are still open.
       v_stmt := 'INSERT INTO ' || prune_running_insts_table 
              || ' SELECT comp.ECID from COMPOSITE_INSTANCE ci, ' 
              || purge_id_table 
              || ' comp, SCA_FLOW_INSTANCE fi  WHERE comp.ECID = ci.ECID AND '
              || ' ci.flow_id = fi.flow_id ';
       if retention_period is not null then
          v_stmt := v_stmt || ' AND (fi.active_component_instances > 0 OR '
                 || ' fi.updated_time >= ' || '''' || retention_period 
                 || '''' || ')';
       else
          v_stmt := v_stmt || ' AND fi.active_component_instances > 0';
       end if;
       EXECUTE immediate v_stmt;
       debug_purge_both
           (prune_running_insts_table, 'Inserted for open composite instances '
           ,write_file);
       COMMIT;

       v_stmt := 'DELETE FROM ' || purge_id_table || ' comp WHERE EXISTS '
              || ' (SELECT 1 FROM ' || prune_running_insts_table || ' run '
              || ' WHERE comp.ECID = run.ECID)';
       EXECUTE immediate v_stmt;
       debug_purge(purge_id_table, 'deleted for open composite instances '
            ,write_file);
       COMMIT;

       soa11g_orabpel.pruneOpenECIDs 
        (purge_id_table,prune_running_insts_table,retention_period,write_file);
       soa11g_mediator.pruneOpenECIDs
        (purge_id_table,prune_running_insts_table,retention_period,write_file);
       soa11g_workflow.pruneOpenECIDs
        (purge_id_table,prune_running_insts_table,retention_period,write_file);
       soa11g_decision.pruneOpenECIDs (
        purge_id_table,prune_running_insts_table,retention_period,write_file);

     end pruneOpenECIDs;

    function deleteNoCompositeIdInstances(min_created_date in timestamp,
                            max_created_date in timestamp,
                            retention_period in timestamp,
                            batch_size in integer,
                            purge_partitioned_component in boolean,
                            componentPartInfo in component_partition_info,
                            stoptime in date,
                            composite_dn in varchar2,
                            soa_partition_name in varchar2,
                            composite_name in varchar2,
                            composite_revision in varchar2,
                            ignore_state in boolean,
                            write_file in utl_file.file_type default null
                            )  return boolean is
    total_rows integer := 0; 
    state integer;
    more_rows_to_delete_all  boolean := false;
    more_rows_to_delete  boolean;
    begin
      
      IF (sysdate >= stoptime) THEN
           return false;
      END IF;

      if purge_partitioned_component = true 
      OR componentPartInfo.bpelPartitioned='N' 
      OR componentPartInfo.bpelPartitioned ='P' 
       then
        log_info_both
          ('calling soa_orabpel.deleteNoCompositeIdInstances',write_file);
        if  soa11g_orabpel.deleteNoCompositeIdInstances( min_created_date,
              max_created_date, retention_period, batch_size, 
              purge_partitioned_component, soa_partition_name, 
              composite_name, composite_revision, ignore_state, write_file) 
         then      
             more_rows_to_delete:=true;
        end if;
        log_info_both
         ('completed soa_orabpel.deleteNoCompositeIdInstances',write_file);
      end if;
   
      IF (sysdate >= stoptime) THEN
           return false;
      END IF;

     if purge_partitioned_component = true 
     OR componentPartInfo.workflowPartitioned='N' 
      then  
        log_info_both
         ('calling workflow.deleteNoCompositeIdInstances',write_file);
         if  soa11g_workflow.deleteNoCompositeIdInstances( min_created_date,
               max_created_date, retention_period, batch_size,
               composite_dn, ignore_state, write_file) 
          then 
             more_rows_to_delete:=true;
         end if;
         log_info_both
          ('completed workflow.deleteNoCompositeIdInstances',write_file);
      end if; 

      IF (sysdate >= stoptime) THEN
           return false;
      END IF;

      if purge_partitioned_component = true 
      OR componentPartInfo.mediatorPartitioned='N' 
      OR componentPartInfo.mediatorPartitioned ='P' then
         log_info_both
          ('calling mediator.deleteNoCompositeIdInstances',write_file);
         if  soa11g_mediator.deleteNoCompositeIdInstances( min_created_date,
               max_created_date, retention_period, batch_size, 
               purge_partitioned_component, composite_dn, ignore_state, 
               write_file) 
          then 
             more_rows_to_delete:=true; 
         end if;
         log_info_both
          ('completed mediator.deleteNoCompositeIdInstances',write_file);
      end if;

      IF (sysdate >= stoptime) THEN
           return false;
      END IF;

      if purge_partitioned_component = true 
      OR componentPartInfo.decisionPartitioned='N' 
       then
         log_info_both
          ('calling decision.deleteNoCompositeIdInstances',write_file);
         if  soa11g_decision.deleteNoCompositeIdInstances( min_created_date,
               max_created_date, retention_period, batch_size, composite_dn,
               ignore_state, write_file) 
        then
            more_rows_to_delete:=true;
        end if;
        log_info_both
         ('completed decision.deleteNoCompositeIdInstances',write_file);
     end if;

     IF (sysdate >= stoptime) THEN
           return false;
      END IF;

     if purge_partitioned_component = true 
     OR componentPartInfo.fabricPartitioned='N' 
     OR componentPartInfo.fabricPartitioned='P' 
      then   
        log_info_both
         ('calling fabric.deleteNoCompositeIdInstances',write_file);
        if  soa11g_fabric.deleteNoCompositeIdInstances(min_created_date,
              max_created_date, retention_period, batch_size, 
              purge_partitioned_component, composite_dn, write_file) 
         then
            more_rows_to_delete:=true;
        end if; 
        log_info_both
         ('completed fabric.deleteNoCompositeIdInstances',write_file);
     end if;

     return more_rows_to_delete;

    exception 
      when others then
        log_error_both('ERROR (deleteNoCompositeIdInstances)',write_file);
        rollback;
        raise;

    end deleteNoCompositeIdInstances;  

   /**
     * procedure delete_composite_instances
     *
     * Deletes the composite instances and all the associated
     * rows in other fabric tables that
     * reference this composite instance directly or indirectly. 
     */    
    procedure delete_composite_instances(instance_id_table in varchar2,
                 purge_partitioned_component in boolean,
                 componentPartInfo in component_partition_info,
                 stoptime in date,
                 ignore_state in boolean,
                 write_file in utl_file.file_type default null
                 )  is   
    begin

     IF (sysdate >= stoptime) THEN
        return;
     END IF;

     if purge_partitioned_component = true 
     OR componentPartInfo.bpelPartitioned='N' 
     OR componentPartInfo.bpelPartitioned='P'
      then
        log_info_both
         ('calling soa_orabpel.deleteComponentInstances',write_file);
        soa11g_orabpel.deleteComponentInstances(instance_id_table,
          purge_partitioned_component,ignore_state,write_file);
        log_info_both
         ('completed soa_orabpel.deleteComponentInstances',write_file);
     end if; 

     IF (sysdate >= stoptime) THEN
        return;
     END IF;

     if purge_partitioned_component = true 
     OR componentPartInfo.workflowPartitioned='N' 
      then   
        log_info_both
          ('calling workflow.deleteComponentInstances',write_file);
        soa11g_workflow.deleteComponentInstances
          (instance_id_table,write_file);
        log_info_both
          ('completed workflow.deleteComponentInstances',write_file);
     end if;      

     IF (sysdate >= stoptime) THEN
         return;
     END IF;
 
     if purge_partitioned_component = true 
     OR componentPartInfo.mediatorPartitioned='N' 
     OR componentPartInfo.mediatorPartitioned='P' 
      then   
        log_info_both
         ('calling mediator.deleteComponentInstances',write_file);
        soa11g_mediator.deleteComponentInstances(instance_id_table,
          purge_partitioned_component,write_file);
        log_info_both
         ('completed mediator.deleteComponentInstances',write_file);
     end if;

     IF (sysdate >= stoptime) THEN
         return;
     END IF;

     if purge_partitioned_component = true 
     OR componentPartInfo.decisionPartitioned='N' 
      then   
        log_info_both
         ('calling decision.deleteComponentInstances',write_file);
        soa11g_decision.deleteComponentInstances
         (instance_id_table,write_file);
        log_info_both
         ('completed decision.deleteComponentInstances',write_file);
     end if;

     IF (sysdate >= stoptime) THEN
         return;
     END IF;

     if purge_partitioned_component = true 
     OR componentPartInfo.fabricPartitioned='N' 
     OR componentPartInfo.fabricPartitioned='P' 
      then   
        log_info_both
          ('calling fabric.deleteComponentInstances',write_file);
        soa11g_fabric.deleteCompositeInstances(instance_id_table,
          purge_partitioned_component,write_file);
        log_info_both
          ('completed fabric.deleteComponentInstances',write_file);
     end if; 

     IF (sysdate >= stoptime) THEN
         return;
     END IF;
      
    exception
      when others then
        log_error_both('ERROR (delete_composite_instances)',write_file);
        rollback;
        raise;

    end delete_composite_instances;
   

   /**
     * procedure delete_stat_alert_tables
     *
     */
    procedure delete_stat_alert_tables(stoptime in date,
                             max_count in integer,
                             batch_size in integer,
                             retention_period in timestamp,
                             write_file in utl_file.file_type default null
                                        )  is


    v_stmt  VARCHAR2(500);
    v_deleted boolean:=true;
    f1_flag boolean:=true;
    f2_flag boolean:=true;
    f3_flag boolean:=true;
    max_count_2 integer := max_count;

    begin

    WHILE  v_deleted  LOOP

      v_deleted := FALSE;

      IF (sysdate >= stoptime) THEN
         EXIT;
      END IF;
      If f1_flag then
         f1_flag := false;
         v_stmt := 'DELETE from BPELNotification '
           || ' where status IN (''SENT'', ''ERROR'') ';
         IF retention_period is not null then
            v_stmt := v_stmt
              || ' and to_timestamp(to_char(CREATEDTIME, ''DD-MON-YYYY HH24:MI:SS''), ''DD-MON-YYYY HH24:MI:SS'') <= ' 
              || '''' || retention_period || '''';
         END IF;
         v_stmt := v_stmt
              || ' and rownum <= ' || batch_size;
         EXECUTE immediate v_stmt;
         IF SQL%FOUND THEN
            v_deleted := true;
            f1_flag := true;
            debug_purge_both('BPELNotification','Deleted ',write_file);
         END IF;
         COMMIT ;
      END IF;

      IF (sysdate >= stoptime) THEN
         EXIT;
      END IF;
      If f2_flag then
         f2_flag := false;
         v_stmt := 'DELETE from BPELNotification_filtermsgs '
           || ' where rownum <= ' || batch_size;
         IF retention_period is not null then
            v_stmt := v_stmt
              || ' and to_timestamp(to_char(RECEIVED_DATE, ''DD-MON-YYYY HH24:MI:SS''), ''DD-MON-YYYY HH24:MI:SS'') <= ' 
              || '''' || retention_period || '''';
         END IF;
         EXECUTE immediate v_stmt;
         IF SQL%FOUND THEN
            v_deleted := true;
            f2_flag := true;
            debug_purge_both('BPELNotification_filtermsgs','Deleted ',write_file);
         END IF;
         COMMIT ;
      END IF;

      IF (sysdate >= stoptime) THEN
         EXIT;
      END IF;
      If f3_flag then
         f3_flag := false;
         v_stmt := 'DELETE from BPELNotification_invaladdrs '
           || ' where rownum <= ' || batch_size;
         IF retention_period is not null then
            v_stmt := v_stmt
              || ' and to_timestamp(to_char(INVALIDATED_TIME, ''DD-MON-YYYY HH24:MI:SS''), ''DD-MON-YYYY HH24:MI:SS'') <= ' 
              || '''' || retention_period || '''';
         END IF;
         EXECUTE immediate v_stmt;
         IF SQL%FOUND THEN
            v_deleted := true;
            f3_flag := true;
            debug_purge_both('BPELNotification_invaladdrs','Deleted ',write_file);
         END IF;
         COMMIT ;
      END IF;

      IF (sysdate >= stoptime) THEN
         EXIT;
      END IF;

    IF max_count > 0 THEN -- a negative max_count means no limit on max_count
      max_count_2 := max_count_2 - batch_size;
      IF max_count_2 <= 0 THEN
         v_deleted := FALSE;
      END If;
    END If;

    END LOOP;

    end delete_stat_alert_tables;

   /**
     * procedure check_purge_session
     *   If another database session is found the purge will exit.
     *   The SID and SERIAL# can be used to kill the session
     *    if appropriate.
     *   The 12c and 11g purges may be able to execute together.
     *     However this is not adivse thus ACTION across 11g and 12c 
     *     purges will be the same but this may change.
     */

    procedure check_purge_session(write_file in utl_file.file_type default null)
    is

    session_found boolean;

    CURSOR c_gvsession is
    select inst_id, sid, serial#
     from gv$session
    where action = 'SOAPURGEEXECUTING'
      and schemaname in
      (SELECT sys_context('USERENV', 'CURRENT_SCHEMA') FROM DUAL);

    begin


       session_found := FALSE;

       FOR gvsess_rec in c_gvsession
       LOOP
          log_info_both
          ('ERROR: Another SOA Purge(parallel/single)is in progress.'
           ,write_file);
          log_info_both('Database gv$session:',write_file);
          log_info_both('  INST_ID  :' || gvsess_rec.inst_id,write_file);
          log_info_both('  SID      :' || gvsess_rec.sid,write_file);
          log_info_both('  SERIAL#  :' || gvsess_rec.serial#,write_file);
          session_found := TRUE;
       END LOOP;

       -- EXIT if another SOA purge (single/parallel) is executing.
       IF session_found THEN
          RAISE_APPLICATION_ERROR
           (-20099, 'ERROR Another SOA Purge(parallel/single)is in progress');
       END IF;

    end check_purge_session;

   /**
     * procedure check_upgrade_session
     *   If the 11g to 12c upgrade is found the purge will exit.
     *   The SID and SERIAL# can be used to kill the session
     *    if appropriate.
     */

    procedure check_upgrade_session(write_file in utl_file.file_type default null) 
    is

    session_found boolean;

    CURSOR c_gvsession is
    select inst_id, sid, serial#
     from gv$session
    where action = 'SOAUPGRADEEXECUTING'
      and schemaname in
      (SELECT sys_context('USERENV', 'CURRENT_SCHEMA') FROM DUAL);

    begin

       session_found := FALSE;

       FOR gvsess_rec in c_gvsession
       LOOP
          log_info_both
          ('ERROR: The 11g to 12c Upgrade is in progress.',write_file);
          log_info_both('Database gv$session:');
          log_info_both('  INST_ID  :' || gvsess_rec.inst_id,write_file);
          log_info_both('  SID      :' || gvsess_rec.sid,write_file);
          log_info_both('  SERIAL#  :' || gvsess_rec.serial#,write_file);
          session_found := TRUE;
       END LOOP;

       -- EXIT if the 11g to 12c upgrade is executing.
       IF session_found THEN
          RAISE_APPLICATION_ERROR
           (-20099, 'ERROR The 11g to 12c Upgrade is in progress');
       END IF;

    end check_upgrade_session;
    
procedure delete_instances ( min_creation_date in timestamp,
                   max_creation_date in timestamp,
                   batch_size in integer default 20000, 
                   -- max_runtime in integer default 60, 
                   stoptime in date default null,
                   retention_period in timestamp default null,
                   purge_partitioned_component in boolean default false,
                   composite_name in  varchar2 default null,
                   composite_revision in varchar2 default null,
                   soa_partition_name in varchar2 default null,
                   keep_workflow_inst in boolean default false,
                   ignore_state in boolean default false,
                   sql_trace IN boolean default false
                   ) is 
    
    composite_id_sql varchar2(4000);
    purge_id_table varchar2(100) := 'ecid_purge';
    -- stoptime date := sysdate + NVL(max_runtime,24*60)/(24*60);
    total_rows integer;
    total_rows_after_pruning integer; 
    orphaned_loop BOOLEAN := TRUE;
    non_orphaned_loop BOOLEAN:=TRUE;
    iterationCount integer :=1;
    deletingOrphanedInstance boolean;
    more_rows_to_delete boolean := false;
    prune_running_insts_table varchar2(100) :='prune_running_insts';
    componentPartInfo component_partition_info;
    v_retention_period timestamp := NVL(retention_period,max_creation_date); 
    composite_dn varchar2(500);
    v_stmt varchar2(500);
    write_file  utl_file.file_type;

    begin

      write_file := get_file11g_main;

      -- Check and exit if another purge is currently running.
      check_purge_session(write_file);
      -- Check and exit if the 11g to 12c upgrade is running.
      check_upgrade_session(write_file);
      -- Set ACTION_NAME on SESSION.
      DBMS_APPLICATION_INFO.SET_MODULE(
            MODULE_NAME => 'SINGLEPURGE11G',
            ACTION_NAME => 'SOAPURGEEXECUTING');

      log_info_both('***************************************',write_file);
      log_info_both('***************************************',write_file);
      log_info_both(' STARTING 11g SINGLE loop PURGE      **',write_file);
      log_info_both('***************************************',write_file);
      log_info_both('***************************************',write_file);
      log_info_both('============================================= Begin SOA Looped Purge ',write_file);
      log_info_both('procedure delete_instances min_creation_date='|| min_creation_date || ' max_creation_date=' || max_creation_date || ' v_retention_period=' || v_retention_period || ' batch_size=' || batch_size || ' stoptime=' || stoptime ,write_file);

      if ignore_state then
         log_info_both('ignore_state is true, data will be purged irrespective of state ');
      end if; 
      log_info_both('time check');
      log_info_both
       ('sysdate  = ' ||TO_CHAR(sysdate,'DD/MON/YYYY:HH24/MI'),write_file);
      log_info_both
       ('stoptime = ' ||TO_CHAR( stoptime,'DD/MON/YYYY:HH24/MI'),write_file);
      IF (sysdate >= stoptime) THEN
            non_orphaned_loop := FALSE;
            orphaned_loop := FALSE;
      END IF;

      if sql_trace then
        v_stmt := 'alter session set max_dump_file_size = unlimited';
        log_info_both(v_stmt,write_file);
        execute immediate v_stmt;
        v_stmt := 'alter session set events '
        || '''10046 trace name context forever, level 12''';
        log_info_both(v_stmt,write_file);
        execute immediate v_stmt;
      end if;

      execute immediate 'delete from SOA_PURGE_LOG where LOG_DATE < sysdate - 60';
      execute immediate 'truncate table ' || prune_running_insts_table;
      execute immediate 'truncate table temp_processed_cikey';
      execute immediate 'truncate table temp_processed_MI';
      execute immediate 'truncate table edn_log_messages';
      componentPartInfo := getComponentPartitionInfo();
      composite_dn := make_composite_dn(soa_partition_name, 
                           composite_name, composite_revision);      
      log_info_both('composite_dn = ' || composite_dn,write_file); 

      WHILE orphaned_loop = TRUE OR non_orphaned_loop = TRUE LOOP

         log_info_both('loop  count = ' || iterationCount,write_file);
         deletingOrphanedInstance := (mod (iterationCount,2)=0);
         IF  deletingOrphanedInstance = false 
         and non_orphaned_loop = true
          then
             log_info_both('deleting non-orphaned instances',write_file);
             composite_id_sql := 
              ' select unique ecid from composite_instance where ' ;
/* For 12c comment out state checking as all ECID will be pruned.
   --------------------------------------------------------------
   if ignore_state = false then
      composite_id_sql :=   composite_id_sql || '(  bitand(state,127)=1 or ';
      composite_id_sql :=   composite_id_sql ||  ' bitand(state,16)=16 or ';
      composite_id_sql :=   composite_id_sql 
             ||  'bitand(state,64)=64 or state between 32 and 63 or ';
      composite_id_sql :=   composite_id_sql 
             ||  'state = 3 or state =19) and  ';
   end if;
*/
             composite_id_sql :=   composite_id_sql 
                              || ' created_time >= ' || '''' 
                              || min_creation_date || ''''; 
             composite_id_sql := composite_id_sql 
                              || ' and created_time <= ' || '''' 
                              || max_creation_date || '''';
             if composite_dn is not  null 
              then 
                composite_id_sql := composite_id_sql 
                                 || ' and composite_dn like ''' 
                                 || composite_dn || '%''';
             end if;
             composite_id_sql := composite_id_sql 
              || ' and ecid  not in (select ecid from prune_running_insts ) ';
             composite_id_sql := ' select * from ( ' || composite_id_sql || ' ) where rownum <= '
               || batch_size ;
             composite_id_sql := 'insert into '||purge_id_table 
               || composite_id_sql;
             execute immediate 'truncate table ' || purge_id_table;
             execute immediate composite_id_sql;
             total_rows := SQL%ROWCOUNT;
             debug_purge_both(purge_id_table,'Inserted = ',write_file);
             commit;
             IF total_rows = 0  THEN
                non_orphaned_loop := FALSE ;
             END IF ;

      /* For 12c comment out state checking as all ECID will be pruned.
         --------------------------------------------------------------
         if ignore_state = false then 
         log_info('perform top-level pruning');
         log_info('total rows before top level pruning = '|| total_rows); 
         toplevelprune(purge_id_table,prune_running_insts_table);
         SELECT count(*) INTO total_rows FROM ecid_purge;
         log_info('total rows after top level pruning = ' || total_rows);
         end if; 
      */
      /*  The pruneOpenEcids should be run on the query again for all cases
         if ignore_state is false since we have eliminated the 
         state check and active_instances check is done in pruneOpenEcids*/

             if ignore_state = false  
              then 
                 log_info_both
                  ('total rows before pruning ' || total_rows,write_file);
                 log_info_both('calling pruneOpenECIDs ',write_file);
                 pruneOpenECIDs (purge_id_table,prune_running_insts_table,
                      v_retention_period,write_file);
                 log_info_both('finished pruneOpenECIDs ',write_file);
                 SELECT count(*) INTO total_rows_after_pruning 
                   FROM ecid_purge;
                 log_info_both
                  ('total_rows_after_pruning ' 
                   || total_rows_after_pruning,write_file);
             end if;

             IF purge_partitioned_component = false 
             AND  ( componentPartInfo.fabricPartitioned='Y' 
                    OR componentPartInfo.fabricPartitioned='P') 
              then
-- since composite instances wouldnt be purged we add the processed ecids to prune_running_inst table
                log_info_both
                 ('inserting processed ecids into prune_running_insts'
                  ,write_file);
                insert into prune_running_insts select ecid from ecid_purge;
                debug_purge_both('prune_running_insts','inserted = ' 
                 ,write_file);
               commit;
             END IF;

             delete_composite_instances(purge_id_table,
               purge_partitioned_component,componentPartInfo, 
               stoptime, ignore_state,write_file);
 
             IF (sysdate >= stoptime) THEN
                non_orphaned_loop := FALSE;
                orphaned_loop := FALSE;
             END IF;
         END IF;
      
         IF  deletingOrphanedInstance = true 
         and orphaned_loop = true then
             log_info_both('deleting orphaned instances',write_file);
             more_rows_to_delete := 
              deleteNoCompositeIdInstances(min_creation_date,
                  max_creation_date, v_retention_period,batch_size,
                  purge_partitioned_component,componentPartInfo,stoptime,
                  composite_dn, soa_partition_name, composite_name, 
                  composite_revision, ignore_state,write_file);
             orphaned_loop := more_rows_to_delete;                
         END IF;
         commit;

         log_info_both('time check',write_file);
         log_info_both
          ('sysdate  = ' ||TO_CHAR(sysdate,'DD/MON/YYYY:HH24/MI'),write_file);
         log_info_both
          ('stoptime = ' ||TO_CHAR( stoptime,'DD/MON/YYYY:HH24/MI'),write_file);
         IF (sysdate >= stoptime) THEN
             non_orphaned_loop := FALSE;
             orphaned_loop := FALSE;
         END IF;
         iterationCount := iterationCount + 1;
      END LOOP;

      if  (sysdate < stoptime 
      and (purge_partitioned_component = true 
           OR soa11g_fabric.getComponentPartitionInfo='N' 
           OR soa11g_fabric.getComponentPartitionInfo='P'))
        then
          soa11g_fabric.deleteRejectedMessages(min_creation_date,
             max_creation_date,batch_size,stoptime,
             purge_partitioned_component,write_file);
      end if;

      /* Note: The delete_stat_alert_tables is used by Single/Parallel purge.
               For single loop the batch_size is passed twice as it
               represents max_count and batch_size.
               The Single Loop purge will only delete the Statistics tables
               if there is time but the Parallel purge will do this in 
               parallel to the spawned threads. 
       */
      delete_stat_alert_tables
           (stoptime,
            -1,
            batch_size,
            v_retention_period,
            write_file);

      execute immediate 'truncate table ' || prune_running_insts_table;
      log_info_both('delete_instances completed successfully',write_file);

      close_file(write_file);
      -- Set ACTION_NAME on SESSION to NULL.
      DBMS_APPLICATION_INFO.SET_MODULE(MODULE_NAME => NULL, 
                            ACTION_NAME => NULL);
 
  EXCEPTION
   when others then
    close_file(write_file);
    DBMS_APPLICATION_INFO.SET_MODULE(MODULE_NAME => NULL, ACTION_NAME => NULL);
    log_error_both('ERROR(delete_instances) ',write_file);     

  end delete_instances; 
