/*
Rem $Header: pcbpel/admin/sqlutils12/soa_purge/oracle/mediator/mediator_createTempTables.sql /st_pcbpel_12.2.1.4.0soabp/2 2020/11/07 09:36:05 apfwkr Exp $
Rem
Rem mediator_createTempTables.sql
Rem
Rem Copyright (c) 2010, 2020, Oracle and/or its affiliates. 
Rem All rights reserved.
Rem
Rem    NAME
Rem      mediator_createTempTables.sql - <one-line expansion of the name>
Rem
Rem    DESCRIPTION
Rem      Mediator purge
Rem
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    apfwkr      11/06/20 - Backport linlxu_bug-32003379_1221-main from main
Rem    apfwkr      11/17/19 - Backport linlxu_bug-30507140_12212-main from main
Rem    sanjain     07/13/10 - Fixing bug 9779594
Rem    sanjain     04/27/10 - Created
Rem
*/

PROCEDURE createTempTables(p_id_table   IN VARCHAR2,
                           max_count in integer,
                           min_creation_date in timestamp,
                           max_creation_date in timestamp,
                           retention_period in timestamp,
                           write_file in utl_file.file_type default null) 
AS
  v_stmt  VARCHAR2(2000);
  v_bound_predicates VARCHAR2(500);
  v_rownum INTEGER;
BEGIN

truncate_temp_tables() ;

v_stmt := 'INSERT into temp_mediator_deferred(msg_id,flow_id,dummy1) ' ||
          'SELECT mdf.msg_id,mdf.flow_id,mdf.dummy1 ' ||
          'from mediator_deferred_message mdf, '
          || p_id_table || ' comp WHERE comp.flow_id = mdf.flow_id';
EXECUTE immediate v_stmt;
debug_purge_both('temp_mediator_deferred', 'Inserted =',write_file);
commit;

v_stmt := 'INSERT into temp_mediator_resequencer(id,flow_id) ' ||
          'SELECT mdr.id,mdr.flow_id ' ||
          'from mediator_resequencer_message mdr, '
          || p_id_table || ' comp WHERE comp.flow_id = mdr.flow_id';
EXECUTE immediate v_stmt;
debug_purge_both('temp_mediator_resequencer', 'Inserted =',write_file);
commit;

/*

The GROUP table is not purged in relation to closed SOA flows. Within 
mediator it pretty much stands alone with a not exists relationship 
to mediator_resequencer_message table.

To reduce the complexity of the purge and improve performance, the 
temp table will be populate in two steps:

1. The initial list of candidate groups will be selected on their relation
to the service table and BOUND by date predicates (min/max/retention).
If not specified then max_count will act as upper bound.

2. ROWNUM is added to result set to aid parallel deletions.

3. The removal of groups that are still in the resequencer table.

*/

v_stmt := 'INSERT into temp_mediator_group(id,row_num) ' ||
          'SELECT mdg.id,rownum ' ||
          'from mediator_group_status mdg, mediator_service_detail mds ' ||
          'where mdg.component_dn = mds.component_dn '||
          'and mdg.level_detail2 = mds.level_detail2 ' ||
          'and mdg.operation = mds.operation ' ||
          'and mdg.status=0 ' ||
          'and (mdg.resequencer_type != ''Standard'' ' ||
               ' or mds.component_status = 1) ';

-- Add date and rownum predicated to BOUND the selection of groups.
v_bound_predicates := null;
if min_creation_date is not null then
   v_bound_predicates := ' and partition_date > ' 
                        || '''' || min_creation_date || '''';
end if;
if max_creation_date is not null then
   v_bound_predicates := v_bound_predicates
                        || ' and partition_date < ' 
                        || '''' || max_creation_date || '''';
end if;
if retention_period is not null then
   v_bound_predicates := v_bound_predicates
                        || ' and (last_received_time < ' 
                        || '''' || retention_period || ''''
                        || ' or (mdg.PARTITION_DATE < ''' || retention_period || ''' AND mdg.RESEQUENCER_TYPE = ''BestEffort'' AND mdg.LOCK_TIME_1 IS NOT NULL))';
end if;
v_bound_predicates := v_bound_predicates 
                     || ' and rownum <= ' || max_count;

v_stmt := v_stmt || v_bound_predicates;
EXECUTE immediate v_stmt;
debug_purge_both('temp_mediator_group', 'Inserted =',write_file);
commit;

/*
 Remove any groups that are still referenced in the resequencer table.
 This does not include the resequencer rows removed in this purge run
 as there is always a chance they might not be removed.
*/
v_stmt := 'DELETE from temp_mediator_group mdg ' ||
          'WHERE exists ' ||
          '(SELECT 1 from mediator_resequencer_message mdr ' ||
          'where mdg.id = mdr.owner_id)';
EXECUTE immediate v_stmt;
debug_purge_both('temp_mediator_group', 'Deleted =',write_file);
commit;


EXCEPTION
  when others then
    log_error_both(v_stmt,write_file);
    raise;

END createTempTables;
