/*
*
* deleteComponentInstancesDOP.sql
*
* Copyright (c) 2010, 2021, Oracle and/or its affiliates. 
*
*    NAME
*      deleteComponentInstancesDOP.sql - BPEL purge script 
*
*
*    DESCRIPTION
*     This script is used to purge BPEL tables provided with a list of composite instance ids
*     in an idTable in a threaded mode.
*
*
*/
/**
* Procedure: createTempTables
*
* Description
* Purpose of this procedure is to create all the required temp tables that
* can be used to purge all of BPEL data that matches the purge criteria.
*
* p_id_table: this is ths table that contrains the purge able composite insance id
*             identified by the calling script.
* p_max_count: Max number of rows to attempt purge
* p_min_creation_date: min date range to attempt purge
* p_max_creation_date: max date range to attempt purge
* p_older_than: BPEL retention period for purge able data
*
*/

PROCEDURE createTempTables(p_id_table   IN VARCHAR2,
                           write_file in utl_file.file_type default null) 
AS
  v_stmt  VARCHAR2(2000);
  v_rownum INTEGER;
  v_cmpst_stmt varchar2(2000) := '';  
BEGIN

truncate_temp_tables() ;

v_stmt := 'INSERT INTO temp_cube_instance ' ||
          'SELECT DISTINCT cikey, ci.flow_id FROM cube_instance ci, '
          ||p_id_table || ' comp WHERE ci.flow_id = comp.flow_id';
EXECUTE immediate v_stmt;
debug_purge_both('temp_cube_instance', 'Inserted =',write_file);

v_stmt := 'Create temp_document_ci_ref' ;
    INSERT INTO temp_document_ci_ref SELECT DISTINCT document_id,t.cikey,t.flow_id
        FROM document_ci_ref d, temp_cube_instance t WHERE d.cikey = t.cikey;
  debug_purge_both('temp_document_ci_ref', 'Inserted = ',write_file);

v_stmt :=  'INSERT INTO temp_document_dlv_msg_ref  SELECT DISTINCT m.message_guid, ddmr.document_id,m.flow_id ' ||
           'FROM document_dlv_msg_ref ddmr, dlv_message m ' ||
           'WHERE ddmr.message_guid = m.message_guid ' ||
           'AND  EXISTS (SELECT 1 FROM ' || p_id_table || ' comp '|| ' where m.flow_id = comp.flow_id)' ;
  EXECUTE immediate v_stmt;
  debug_purge_both('temp_document_dlv_msg_ref', 'Inserted = ',write_file);

v_stmt := 'create temp_xml_document';
    INSERT INTO temp_xml_document SELECT tpic.document_id,tpic.flow_id FROM temp_document_dlv_msg_ref tpic; 
    COMMIT;
    INSERT INTO temp_xml_document select tpic.document_id,tpic.flow_id FROM temp_document_ci_ref tpic;
    COMMIT;
    DELETE FROM temp_xml_document txd1 WHERE txd1.rowid > ANY (SELECT txd2.rowid FROM temp_xml_document txd2 WHERE txd1.document_id = txd2.document_id);
    COMMIT;
EXCEPTION
  when others then
    log_error_both(v_stmt,write_file);
    raise;

END createTempTables;
/*
* Procedure: deleteComponentInstaces
*
* Description
* This procedure purges following BPEL tables based on temp tables created in earlier step.
*
* Note: You need to obtain EXECUTE privilege to run this procedure.
* GRANT EXECUTE ON dbms_lock TO user_name
*
*/

FUNCTION deleteComponentInstances(p_dop NUMBER, p_thread NUMBER, 
                     p_batch_size NUMBER default 20000,
                     p_stop_time DATE default NULL,
                     write_file in utl_file.file_type default NULL, 
                     purge_partitioned_component in boolean default false) 
RETURN BOOLEAN
AS
  v_stmt  VARCHAR2(2000);
  v_deleted boolean:=true;
  v_counter NUMBER := 1;
  dlv_mds_ref_flag boolean:=true;
  dlv_mds_flag boolean:=true;
  cursor_flag boolean:=true;
  v_stoptime date :=p_stop_time; 

  -- Added for cursor optimization
  l_num_rows number := 0;
  TYPE purge_num_type IS TABLE OF NUMBER;
  l_cube_instances purge_num_type;

  CURSOR c_temp_cube_instance_thread IS
  SELECT tpic.cikey FROM temp_cube_instance tpic 
    WHERE mod (tpic.flow_id, p_dop)=p_thread;


BEGIN

log_info('=== Begin BPEL purge for thread = ' || p_thread,write_file);
log_info('BPEL batch size = ' || p_batch_size || ' for thread = ' || p_thread,write_file);

-- purge HEADERS_PROPERTIES before DOCUMENT_DLV_MSG_REF table to avoid dangling records
IF (sysdate < v_stoptime) THEN
    v_deleted := true;
    v_counter := 1;
    while  v_deleted  LOOP -- Purge HEADERS_PROPERTIES table
        v_deleted := false;
        v_stmt := 'Purge HEADERS_PROPERTIES' ;
            DELETE FROM HEADERS_PROPERTIES WHERE message_guid IN (SELECT tddmr.message_guid 
                FROM  temp_document_dlv_msg_ref tddmr WHERE
                mod (dbms_utility.get_hash_value(tddmr.message_guid,0,p_dop), p_dop)=p_thread
                )
                AND rownum <= p_batch_size;
            IF SQL%FOUND THEN
                v_deleted := true;
                log_info('HEADERS_PROPERTIES purge loop ' || v_counter || ' for thread = ' || p_thread,write_file);
                debug_purge('HEADERS_PROPERTIES',null,write_file);
                COMMIT;
            END IF;
        v_counter := v_counter + 1;
    END LOOP;
END IF;

-- purge XML_DOCUMENT before DOCUMENT_DLV_MSG_REF table to avoid dangling records 
IF (sysdate < v_stoptime) THEN
    v_deleted := true;
    v_counter := 1;
    while  v_deleted  LOOP -- Purge XML_DOCUMENT table
        v_deleted := false;
        v_stmt := 'Purge XML_DOCUMENT';
        if purge_partitioned_component = true OR is_table_partitioned('XML_DOCUMENT','DOC_PARTITION_DATE') = false then
            DELETE FROM  XML_DOCUMENT d WHERE d.document_id IN (
                SELECT tpic.document_id FROM temp_xml_document tpic 
                WHERE (mod (tpic.flow_id, p_dop)=p_thread)) 
                AND rownum < p_batch_size;
            IF SQL%FOUND THEN
                v_deleted := true;
                log_info('XML_DOCUMENT purge loop ' || v_counter || ' for thread = ' || p_thread,write_file);
                debug_purge('XML_DOCUMENT',null,write_file);
                COMMIT ;
            END IF;
        END IF;
        v_counter := v_counter + 1;
    END LOOP;
END IF;

-- Due to temp_document_dlv_msg_ref is populated using DOCUMENT_DLV_MSG_REF and DLV_MESSAGE tables
-- purge both tables in the same loop to avoid dangling records
IF (sysdate < v_stoptime) THEN
    v_deleted := true;
    v_counter := 1;
    while  v_deleted  LOOP
        v_deleted := false;
        v_stmt := 'Purge DOCUMENT_DLV_MSG_REF' ;  -- Purge DOCUMENT_DLV_MSG_REF table
            If dlv_mds_ref_flag THEN
                dlv_mds_ref_flag := false;
                DELETE FROM DOCUMENT_DLV_MSG_REF d WHERE d.document_id IN ( 
                    SELECT tpic.document_id FROM temp_document_dlv_msg_ref tpic 
                    WHERE (mod (tpic.flow_id, p_dop)=p_thread)) 
                    AND rownum <= p_batch_size;
                IF SQL%FOUND THEN
                    v_deleted := true;
                    dlv_mds_ref_flag := true;
                    log_info('DOCUMENT_DLV_MSG_REF purge loop ' || v_counter || ' for thread = ' || p_thread,write_file);
                    debug_purge('DOCUMENT_DLV_MSG_REF',null,write_file);
                    COMMIT ;
                END IF;
            END IF;

        v_stmt := 'Purge DLV_MESSAGE' ;  -- Purge DLV_MESSAGE table
            If dlv_mds_flag THEN
                dlv_mds_flag := false;
                DELETE FROM DLV_MESSAGE m where message_guid in (
                    select tddmr.message_guid from  temp_document_dlv_msg_ref tddmr where  
                    mod (dbms_utility.get_hash_value(tddmr.message_guid,0,p_dop), p_dop)=p_thread
                    )
                AND rownum <= p_batch_size;
                IF SQL%FOUND THEN
                    v_deleted := true;
                    dlv_mds_flag := true;
                    log_info('DLV_MESSAGE purge loop ' || v_counter || ' for thread = ' || p_thread,write_file);
                    debug_purge('DLV_MESSAGE',null,write_file);
                    COMMIT ;
                END IF;
            END IF;
            v_counter := v_counter + 1;

    END LOOP;
END IF;

OPEN c_temp_cube_instance_thread;

-- Bug 25379310 - to avoid dangling records purge CUBE_INSTANCE table last
IF (sysdate < v_stoptime) THEN
    v_deleted := true;
    v_counter := 1;
    while  v_deleted  LOOP -- this loop is for keeping transactions short

        v_deleted := false;
        log_info('Begin main bpel purge loop ' || v_counter || ' for thread = ' || p_thread,write_file);

        FETCH c_temp_cube_instance_thread BULK COLLECT into l_cube_instances LIMIT p_batch_size;
        l_num_rows := l_cube_instances.count;
        log_info('cube instances count = ' || l_num_rows, write_file);

        v_stmt := 'Purge AG_INSTANCE' ;
            IF cursor_flag then
                FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
                    DELETE FROM AG_INSTANCE WHERE cikey = l_cube_instances(i);
                debug_purge('AG_INSTANCE',null,write_file);
                COMMIT;
            END IF;

        v_stmt := 'Purge TEST_DETAILS' ;
            IF cursor_flag then
                FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
                    DELETE FROM TEST_DETAILS WHERE cikey = l_cube_instances(i);
                debug_purge('TEST_DETAILS',null,write_file);
                COMMIT;
            END IF;
    
        v_stmt := 'Purge CUBE_SCOPE' ;
        IF purge_partitioned_component = true OR is_table_partitioned('CUBE_SCOPE','CI_PARTITION_DATE') = false then
            IF cursor_flag then
                FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
                   DELETE FROM CUBE_SCOPE WHERE cikey = l_cube_instances(i);
                debug_purge('CUBE_SCOPE',null,write_file);
                COMMIT;
            END IF;
        END IF;

        v_stmt := 'Purge CI_INDEXES' ;
            IF cursor_flag then
                FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
                   DELETE FROM CI_INDEXES WHERE cikey = l_cube_instances(i);
                debug_purge('CI_INDEXES',null,write_file);
                COMMIT;
            END IF;

        v_stmt := 'Purge WORK_ITEM' ;
        IF cursor_flag then
            FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
                DELETE FROM WORK_ITEM WHERE cikey = l_cube_instances(i);
            debug_purge('WORK_ITEM',null,write_file);
            COMMIT ;
        END IF;

        v_stmt := 'Purge WI_FAULT' ;
        IF cursor_flag then
            FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
                DELETE FROM WI_FAULT WHERE cikey = l_cube_instances(i);
            debug_purge('WI_FAULT',null,write_file);
            COMMIT ;
        END IF;

        v_stmt := 'Purge DOCUMENT_CI_REF' ;
        IF cursor_flag then
            FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
                DELETE FROM DOCUMENT_CI_REF WHERE cikey = l_cube_instances(i);
            debug_purge('DOCUMENT_CI_REF',null,write_file);
            COMMIT ;
        END IF;

        v_stmt := 'Purge DLV_SUBSCRIPTION';
        IF cursor_flag then
            FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
                DELETE FROM DLV_SUBSCRIPTION WHERE cikey = l_cube_instances(i);
            debug_purge('DLV_SUBSCRIPTION',null,write_file);
            COMMIT;
        END IF;

        v_stmt := 'Purge DLV_AGGREGATION';
        IF cursor_flag then
            FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
                DELETE FROM DLV_AGGREGATION WHERE cikey = l_cube_instances(i);
            debug_purge('DLV_AGGREGATION',null,write_file);
            COMMIT;
        END IF;

-- MJB 
IF purge_partitioned_component = true
OR is_table_partitioned('VARIABLE_SENSOR_VALUES','CREATION_DATE') = false then
   v_stmt := 'Purge VARIABLE_SENSOR_VALUES';
       IF cursor_flag then
           FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
              DELETE FROM VARIABLE_SENSOR_VALUES
               WHERE PROCESS_INSTANCE_ID = l_cube_instances(i);
           debug_purge('VARIABLE_SENSOR_VALUES',null,write_file);
           COMMIT;
       END IF;
END IF;

IF purge_partitioned_component = true
OR is_table_partitioned('FAULT_SENSOR_VALUES','CREATION_DATE') = false then
   v_stmt := 'Purge FAULT_SENSOR_VALUES';
       IF cursor_flag then
           FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
              DELETE FROM FAULT_SENSOR_VALUES
               WHERE PROCESS_INSTANCE_ID = l_cube_instances(i);
           debug_purge('FAULT_SENSOR_VALUES',null,write_file);
           COMMIT;
       END IF;
END IF;

IF purge_partitioned_component = true
OR is_table_partitioned('ACTIVITY_SENSOR_VALUES','CREATION_DATE') = false then
   v_stmt := 'Purge ACTIVITY_SENSOR_VALUES';
       IF cursor_flag then
           FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
              DELETE FROM ACTIVITY_SENSOR_VALUES
               WHERE PROCESS_INSTANCE_ID = l_cube_instances(i);
           debug_purge('ACTIVITY_SENSOR_VALUES',null,write_file);
           COMMIT;
       END IF;
END IF;

v_stmt := 'Purge CUBE_INSTANCE';
    IF purge_partitioned_component = true OR is_table_partitioned('CUBE_INSTANCE','CPST_INST_CREATED_TIME') = false then
    IF cursor_flag then
        FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
           DELETE FROM CUBE_INSTANCE WHERE cikey = l_cube_instances(i);
        debug_purge('CUBE_INSTANCE',null,write_file);
        commit; 
    END IF;
    END IF;
    
        cursor_flag := false;
        if l_num_rows = p_batch_size then
            cursor_flag := true;
            v_deleted := true;
        end if;  
        v_counter := v_counter + 1;

    END LOOP;
END IF;

log_info('=== END BPEL purge for thread = ' || p_thread,write_file);
CLOSE c_temp_cube_instance_thread;
return TRUE;

EXCEPTION
    when others then
       log_error(v_stmt, write_file);
    CLOSE c_temp_cube_instance_thread;
    raise;

END deleteComponentInstances;

