/*
*
* deleteComponentInstancesDOP.sql
*
* Copyright (c) 2010, 2020, Oracle and/or its affiliates. All rights reserved.
*
*    NAME
*      deleteComponentInstancesDOP.sql - BPEL purge script 
*
*
*    DESCRIPTION
*     This script is used to purge b2B tables provided with a list of composite instance ids
*     in an idTable in a threaded mode.
*
*
*/
/**
* Procedure: createTempTables
*
* Description
* Purpose of this procedure is to create all the required temp tables that
* can be used to purge all of B2B data that matches the purge criteria.
*
* p_id_table: this is ths table that contrains the purge able composite insance id
*             identified by the calling script.
* p_max_count: Max number of rows to attempt purge
* p_min_creation_date: min date range to attempt purge
* p_max_creation_date: max date range to attempt purge
* p_older_than: Not used by B2B
*
*/

PROCEDURE createTempTables(p_id_table   IN VARCHAR2, 
                      p_max_count IN INTEGER,
                      p_min_creation_date in timestamp,
                      p_max_creation_date in timestamp,
                      p_older_than TIMESTAMP,
                      write_file in utl_file.file_type default NULL)
AS
  v_stmt  VARCHAR2(2000);
  v_rownum INTEGER;
BEGIN

truncate_temp_tables() ;
  v_stmt := 'insert into b2b_purge_temp_bm(bm_id, bm_message_id, bm_payload_storage, bm_attachment_storage, bm_native_data_storage, bm_translated_data_storage, wm_packed_message, wm_payload_storage, am_payload, am_attachment_storage, ext_bm_id, am_id, wm_id)        SELECT bm.id, bm.message_id, bm.payload_storage, bm.attachment_storage, bm.native_data_storage, 
       	      bm.translated_data_storage,
       	      wm.packed_message, wm.payload_storage,  
	      am.payload, am.attachment_storage, ext.id, am.id, wm.id
       FROM b2b_business_message bm, b2b_wire_message wm, b2b_app_message am, b2b_ext_business_message ext, ' || p_id_table || ' temp_table
       WHERE am.fabric_ecid = temp_table.ECID
       AND bm.wire_message = wm.id (+)
       AND bm.ext_business_message = ext.id (+)
       AND bm.app_message = am.id
       AND ROWNUM <= ' || p_max_count;
  EXECUTE immediate v_stmt;

  v_rownum := p_max_count - SQL%ROWCOUNT ;
  debug_purge_both('b2b_purge_temp_bm', 'Inserted =',write_file);
IF v_rownum > 0 THEN

  v_stmt := 'insert into b2b_purge_temp_bm(bm_id, bm_message_id, bm_payload_storage, bm_attachment_storage, bm_native_data_storage, bm_translated_data_storage, wm_packed_message, wm_payload_storage, am_payload, am_attachment_storage, ext_bm_id, am_id, wm_id)        SELECT bm.id, bm.message_id, bm.payload_storage, bm.attachment_storage, bm.native_data_storage, 
       	      bm.translated_data_storage,
       	      wm.packed_message, wm.payload_storage,  
	      am.payload, am.attachment_storage, ext.id, am.id, wm.id
       FROM b2b_business_message bm, b2b_wire_message wm, b2b_app_message am, b2b_ext_business_message ext
       WHERE am.FABRIC_COMPOSITE_INSTANCE_ID is null
       AND bm.CPST_INST_CREATED_TIME BETWEEN ' || p_min_creation_date || 
     ' AND ' || p_max_creation_date || ' 
       AND (bm.state = ''MSG_COMPLETE'' OR bm.state = ''MSG_ERROR'')
       AND bm.wire_message = wm.id (+)
       AND bm.ext_business_message = ext.id (+)
       AND bm.app_message = am.id (+)
       AND ROWNUM <= ' || v_rownum;
  EXECUTE immediate v_stmt;
  debug_purge_both('b2b_purge_temp_bm', 'Inserted = ',write_file);
  COMMIT;
END IF;

EXCEPTION
  when others then
    log_error_both(v_stmt,write_file);
    raise;

END createTempTables;
/*
* Procedure: deleteComponentInstaces
*
* Note: You need to obtain EXECUTE privilege to run this procedure.
* GRANT EXECUTE ON dbms_lock TO user_name
*
*/

FUNCTION deleteComponentInstances(p_dop NUMBER, p_thread NUMBER, 
                           p_batch_size NUMBER default 20000,
                           p_stop_time DATE default NULL,
                           write_file in utl_file.file_type default NULL)
 RETURN BOOLEAN
AS
   CURSOR b2b_cur IS
     select bm_message_id, bm_payload_storage, bm_attachment_storage, bm_native_data_storage, bm_translated_data_storage, wm_packed_message, wm_payload_storage, am_payload, am_attachment_storage from b2b_purge_temp_bm;

  TYPE bm_id_type IS TABLE OF b2b_purge_temp_bm.bm_message_id%TYPE;
  bm_id bm_id_type;
  TYPE bm_ps_type IS TABLE OF b2b_purge_temp_bm.bm_payload_storage%TYPE;
  bm_ps bm_ps_type;
  TYPE bm_as_type IS TABLE OF b2b_purge_temp_bm.bm_attachment_storage%TYPE;
  bm_as bm_as_type;
  TYPE bm_ns_type IS TABLE OF b2b_purge_temp_bm.bm_native_data_storage%TYPE;
  bm_ns bm_ns_type;
  TYPE bm_ts_type IS TABLE OF b2b_purge_temp_bm.bm_translated_data_storage%TYPE;
  bm_ts bm_ts_type;
  TYPE wm_pm_type IS TABLE OF b2b_purge_temp_bm.wm_packed_message%TYPE;
  wm_pm wm_pm_type;
  TYPE wm_ps_type IS TABLE OF b2b_purge_temp_bm.wm_payload_storage%TYPE;
  wm_ps wm_ps_type;
  TYPE am_p_type IS TABLE OF b2b_purge_temp_bm.am_payload%TYPE;
  am_p am_p_type;
  TYPE am_as_type IS TABLE OF b2b_purge_temp_bm.am_attachment_storage%TYPE;
  am_as am_as_type;

  v_stmt  VARCHAR2(2000);
  v_deleted boolean:=true;
  v_counter NUMBER := 1;
  f1_flag boolean:=true;
  v_stoptime date :=p_stop_time; 
BEGIN

v_deleted := true;
OPEN b2b_cur;
while  v_deleted  LOOP -- this loop is for keeping transactions short

    v_deleted := false;
    log_info('Begin B2B purge loop ' || v_counter || ' for thread = ' || p_thread,write_file);

v_stmt := 'Purge B2B_BUSINESS_MESSAGE' ;
    IF f1_flag then
    f1_flag:=false;

     FETCH b2b_cur BULK COLLECT INTO bm_id, bm_ps, bm_as, bm_ns, bm_ts, wm_pm, wm_ps, am_p, am_as LIMIT p_batch_size;
      dbms_output.put_line('TOTAL DOP BM_ID = ' || bm_id.count());

     FORALL i IN bm_ps.first .. bm_ps.last
        DELETE FROM b2b_data_storage WHERE id = bm_ps(i) AND mod (dbms_utility.get_hash_value(bm_ps(i),0,p_dop), p_dop)=p_thread;

     IF SQL%FOUND THEN
            v_deleted := true;
            f1_flag:=true;
     END IF;

     FORALL i IN bm_as.first .. bm_as.last
        DELETE FROM b2b_data_storage WHERE id = bm_as(i) AND mod (dbms_utility.get_hash_value(bm_as(i),0,p_dop), p_dop)=p_thread;

     FORALL i IN bm_ns.first .. bm_ns.last
        DELETE FROM b2b_data_storage WHERE id = bm_ns(i) AND mod (dbms_utility.get_hash_value(bm_ns(i),0,p_dop), p_dop)=p_thread;

     FORALL i IN bm_ts.first .. bm_ts.last
        DELETE FROM b2b_data_storage WHERE id = bm_ts(i) AND mod (dbms_utility.get_hash_value(bm_ts(i),0,p_dop), p_dop)=p_thread;

     FORALL i IN wm_pm.first .. wm_pm.last
        DELETE FROM b2b_data_storage WHERE id = wm_pm(i) AND mod (dbms_utility.get_hash_value(wm_pm(i),0,p_dop), p_dop)=p_thread;

     FORALL i IN wm_ps.first .. wm_ps.last
        DELETE FROM b2b_data_storage WHERE id = wm_ps(i) AND mod (dbms_utility.get_hash_value(wm_ps(i),0,p_dop), p_dop)=p_thread;

     FORALL i IN am_p.first .. am_p.last
        DELETE FROM b2b_data_storage WHERE id = am_p(i) AND mod (dbms_utility.get_hash_value(am_p(i),0,p_dop), p_dop)=p_thread;

     FORALL i IN am_as.first .. am_as.last
        DELETE FROM b2b_data_storage WHERE id = am_as(i) AND mod (dbms_utility.get_hash_value(am_as(i),0,p_dop), p_dop)=p_thread;

    END IF;
    debug_purge('b2b_purge_temp_bm',null,write_file);
    COMMIT;

    v_counter := v_counter + 1;

    -- exit loop if out of ime
    IF (sysdate >= v_stoptime) THEN
       v_deleted := FALSE;
       CLOSE b2b_cur;
       return FALSE ;
    END IF;
END LOOP;
    CLOSE b2b_cur;
    return TRUE;

EXCEPTION
  when others then
    log_error(v_stmt, write_file);
    CLOSE b2b_cur;
    raise;
END deleteComponentInstances;
