/*
*
* deleteComponentInstancesDOP.sql
*
* Copyright (c) 2010, 2021, Oracle and/or its affiliates. 
*
*    NAME
*      deleteComponentInstancesDOP.sql - BPEL purge script 
*
*
*    DESCRIPTION
*     This script is used to purge BPEL tables provided with a list of composite instance ids
*     in an idTable in a threaded mode.
*
*
*/
/**
* Procedure: createTempTables
*
* Description
* Purpose of this procedure is to create all the required temp tables that
* can be used to purge all of BPEL data that matches the purge criteria.
*
* p_id_table: this is ths table that contrains the purge able composite insance id
*             identified by the calling script.
* p_max_count: Max number of rows to attempt purge
* p_min_creation_date: min date range to attempt purge
* p_max_creation_date: max date range to attempt purge
* p_older_than: BPEL retention period for purge able data
*
*/

PROCEDURE createTempTables(p_id_table   IN VARCHAR2, 
                           p_max_count IN INTEGER,
                           p_min_creation_date in timestamp,
                           p_max_creation_date in timestamp,
                           p_older_than TIMESTAMP,
                           composite_name in varchar2,
                           composite_revision in varchar2,
                           soa_partition_name in varchar2,
                           ignore_composite in boolean default false,
                           write_file in utl_file.file_type default null
						   )
AS
  v_stmt  VARCHAR2(2000);
  v_rownum INTEGER;
  v_cmpst_stmt varchar2(2000) := '';  
BEGIN

        if ignore_composite then 
            log_info('inside orabpel  createTempTables ignore_composite is true',write_file);
        else
            log_info('inside orabpel  createTempTables ignore_composite is false',write_file);
        end if;
truncate_temp_tables() ;

-- Add all ciKeys that have a composite instance id
-- p_idTable has the pruned list, so add all to the temp table
v_stmt := 'INSERT INTO temp11g_cube_instance ' ||
              'SELECT cikey, ci.ecid FROM cube_instance ci, '||p_id_table || ' 
               comp WHERE ci.ecid = comp.ecid';
EXECUTE immediate v_stmt;
-- if we still have space, all the no composite instance id rows too
-- All other BPEL temp tables are created based off temp11g_cube_instance
v_rownum := p_max_count - SQL%ROWCOUNT ;
debug_purge('temp11g_cube_instance', 'Inserted =');
COMMIT;
IF v_rownum > 0 THEN
	if soa_partition_name is not null then 
	   v_cmpst_stmt := ' AND domain_name = ''' || soa_partition_name || '''';
    end if;	   
	if  composite_name is not null then
	   v_cmpst_stmt := v_cmpst_stmt || ' AND composite_name = ''' || composite_name ||'''';
	end if;
	if  composite_name is not null and composite_revision is not null then
	   v_cmpst_stmt := v_cmpst_stmt || ' AND composite_revision = ''' || composite_revision ||'''' ;
	end if;
	if  ignore_composite = false then
	   v_cmpst_stmt := v_cmpst_stmt || ' AND CMPST_ID is null' ;
	end if;
    EXECUTE IMMEDIATE 'INSERT INTO temp11g_cube_instance(CIKEY, ECID) ' || 
         'SELECT CIKEY, ECID FROM CUBE_INSTANCE ci WHERE STATE >= 5 AND MODIFY_DATE < '||
         ''''|| p_older_than ||''''|| ' AND   CPST_INST_CREATED_TIME  BETWEEN '||'''' ||  
         p_min_creation_date ||''''|| '  AND ' ||''''|| p_max_creation_date  ||''''||
         v_cmpst_stmt ||
         ' AND NOT EXISTS (SELECT 1 FROM DLV_MESSAGE dlv WHERE dlv.STATE IN (0,1) AND dlv.ECID=ci.ECID) ' ||
         ' AND ROWNUM <= ' || v_rownum ;
    debug_purge_both('temp11g_cube_instance', 'Inserted = ',write_file);
    COMMIT;
END IF ;

v_stmt := 'Create temp11g_document_ci_ref' ;
    INSERT INTO temp11g_document_ci_ref SELECT document_id,t.cikey, t.ecid
        FROM document_ci_ref d, temp11g_cube_instance t WHERE d.cikey = t.cikey;
    debug_purge_both('temp11g_document_ci_ref', 'Inserted = ',write_file);
    COMMIT;

v_stmt := 'Create temp11g_document_dlv_msg_ref from dlv_message type = 1 or 2';
    INSERT INTO temp11g_document_dlv_msg_ref  SELECT m.message_guid, ddmr.document_id,t.cikey,t.ecid
           FROM document_dlv_msg_ref ddmr, dlv_message m , temp11g_cube_instance t
           WHERE ddmr.message_guid = m.message_guid
               AND m.state in (2,3) AND  m.cikey = t.cikey ; 

v_rownum := p_max_count - SQL%ROWCOUNT ;
    debug_purge_both('temp11g_document_dlv_msg_ref', 'Inserted = ',write_file);
    COMMIT;

IF v_rownum > 0 THEN
    -- Still have more space so include m.cikey = 0 criteria  also. 
    
    EXECUTE IMMEDIATE 'INSERT INTO temp11g_document_dlv_msg_ref (message_guid,document_id) '||
            'SELECT m.message_guid,ddmr.document_id ' ||
               'FROM document_dlv_msg_ref ddmr, dlv_message m  ' ||
               'WHERE ddmr.message_guid = m.message_guid ' ||
                        'AND m.state in (2,3) AND m.cikey = 0  and ROWNUM <= ' ||v_rownum ;
    COMMIT;

END IF;

v_stmt := 'create temp11g_xml_document';
    INSERT INTO temp11g_xml_document SELECT tpic.document_id,tpic.cikey,tpic.ecid FROM temp11g_document_dlv_msg_ref tpic; 
    COMMIT;
    INSERT INTO temp11g_xml_document select tpic.document_id,tpic.cikey,tpic.ecid FROM temp11g_document_ci_ref tpic;
    COMMIT;
    DELETE FROM temp11g_xml_document doc WHERE doc.document_id IN (select b2b.document_id from B2B_DATA_STORAGE b2b);
    COMMIT;
    DELETE FROM temp11g_xml_document txd1 WHERE txd1.rowid > ANY (SELECT txd2.rowid FROM temp11g_xml_document txd2 WHERE txd1.document_id = txd2.document_id);
    COMMIT;
EXCEPTION
  when others then
    log_error_both(v_stmt,write_file);
    raise;

END createTempTables;
/*
* Procedure: deleteComponentInstaces
*
* Description
* This procedure purges following BPEL tables based on temp tables created in earlier step.
*
*    HEADERS_PROPERTIES
*    AG_INSTANCE
*    TEST_DETAILS
*    CUBE_SCOPE
*    AUDIT_COUNTER
*    AUDIT_TRAIL
*    AUDIT_DETAILS
*    CI_INDEXS
*    WORK_ITEM
*    WI_FAULT
*    XML_DOCUMENT
*    DOCUMENT_DLV_MSG_REF
*    DOCUMENT_CI_REF
*    DLV_MESSAGE
*    DLV_SUBSCRIPTION
*    CUBE_INSTANCE
*
*    BPM_AUDIT_QUERY
*    BPM_MEASUREMENT_ACTIONS
*    BPM_MEASUREMENT_ACTION_EXCEPS
*    BPM_CUBE_AUDITINSTANCE
*    BPM_CUBE_TASKPERFORMANCE
*    BPM_CUBE_PROCESSPERFORMANCE
*    
*
* Note: You need to obtain EXECUTE privilege to run this procedure.
* GRANT EXECUTE ON dbms_lock TO user_name
*
*/

FUNCTION deleteComponentInstances(p_dop NUMBER, p_thread NUMBER, 
                       p_batch_size NUMBER default 20000,
                       p_stop_time DATE default NULL,
                       write_file in utl_file.file_type default NULL,
                       purge_partitioned_component in boolean default false)
 RETURN BOOLEAN
AS
  v_stmt  VARCHAR2(2000);
  v_deleted boolean:=true;
  v_counter NUMBER := 1;
  f1_flag boolean:=true;
  f2_flag boolean:=true;
  f3_flag boolean:=true;
  f4_flag boolean:=true;
  cursor_flag boolean:=true;
  v_stoptime date :=p_stop_time; 

  -- Added for cursor optimization
  l_num_rows number := 0;
  TYPE purge_num_type IS TABLE OF NUMBER;
  l_cube_instances purge_num_type;

  CURSOR c_temp_cube_instance_thread IS
  SELECT tpic.cikey FROM temp11g_cube_instance tpic 
    WHERE tpic.ecid IS NULL AND mod (tpic.cikey, p_dop)=p_thread
    OR (tpic.ecid IS NOT NULL and mod (dbms_utility.get_hash_value(tpic.ecid,0,p_dop), p_dop)=p_thread);


BEGIN

OPEN c_temp_cube_instance_thread;

v_deleted := true;
while  v_deleted  LOOP -- this loop is for keeping transactions short

    v_deleted := false;
    log_info('Begin BPEL purge loop ' || v_counter || ' for thread = ' || p_thread,write_file);

    FETCH c_temp_cube_instance_thread BULK COLLECT into l_cube_instances LIMIT p_batch_size;
    l_num_rows := l_cube_instances.count;
    log_info('cube instances count = ' || l_num_rows, write_file);

v_stmt := 'Purge HEADERS_PROPERTIES' ;
    IF f1_flag then
        f1_flag:=false;
        DELETE FROM HEADERS_PROPERTIES WHERE message_guid IN (SELECT tddmr.message_guid 
           FROM  temp11g_document_dlv_msg_ref tddmr WHERE
           mod (dbms_utility.get_hash_value(tddmr.message_guid,0,p_dop), p_dop)=p_thread
          )
          AND rownum <= p_batch_size;
        IF SQL%FOUND THEN
            v_deleted := true;
            f1_flag:=true;
            debug_purge('HEADERS_PROPERTIES',null,write_file);
            COMMIT;
        END IF;
    END IF;

v_stmt := 'Purge AG_INSTANCE' ;
    IF cursor_flag then
        FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
          DELETE FROM AG_INSTANCE WHERE cikey = l_cube_instances(i);
          debug_purge('AG_INSTANCE',null,write_file);
          COMMIT;
    END IF;

v_stmt := 'Purge TEST_DETAILS' ;
    IF cursor_flag then
        FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
        DELETE FROM TEST_DETAILS WHERE cikey = l_cube_instances(i);
        debug_purge('TEST_DETAILS',null,write_file);
        COMMIT;
    END IF;
    
v_stmt := 'Purge CUBE_SCOPE' ;
    IF purge_partitioned_component = true OR is_table_partitioned('CUBE_SCOPE','CI_PARTITION_DATE') = false then
    IF cursor_flag then
       FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
       DELETE FROM CUBE_SCOPE WHERE cikey = l_cube_instances(i);
       debug_purge('CUBE_SCOPE',null,write_file);
       COMMIT;
    END IF;
	END IF;

v_stmt := 'Purge AUDIT_COUNTER' ;
    IF cursor_flag then
       FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
          DELETE FROM AUDIT_COUNTER WHERE cikey = l_cube_instances(i);
          debug_purge('AUDIT_COUNTER',null,write_file);
          COMMIT;
    END IF;

v_stmt := 'Purge AUDIT_TRAIL' ;
    IF cursor_flag then
        FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
          DELETE FROM AUDIT_TRAIL WHERE cikey = l_cube_instances(i);
          debug_purge('AUDIT_TRAIL',null,write_file);
          COMMIT;
    END IF;

v_stmt := 'Purge AUDIT_DETAILS' ;
    IF cursor_flag then
        FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
          DELETE FROM AUDIT_DETAILS WHERE cikey = l_cube_instances(i);
        debug_purge('AUDIT_DETAILS',null,write_file);
        COMMIT;
    END IF;

v_stmt := 'Purge CI_INDEXS' ;
    IF cursor_flag then
        FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
           DELETE FROM CI_INDEXES WHERE cikey = l_cube_instances(i);
         debug_purge('CI_INDEXES',null,write_file);
         COMMIT;
    END IF;

v_stmt := 'Purge WORK_ITEM' ;
    IF cursor_flag then
        FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
           DELETE FROM WORK_ITEM WHERE cikey = l_cube_instances(i);
        debug_purge('WORK_ITEM',null,write_file);
        COMMIT ;
    END IF;

v_stmt := 'Purge WI_FAULT' ;
    IF cursor_flag then
        FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
           DELETE FROM WI_FAULT WHERE cikey = l_cube_instances(i);
        debug_purge('WI_FAULT',null,write_file);
        COMMIT ;
    END IF;

v_stmt := 'Purge XML_DOCUMENT';
    if purge_partitioned_component = true OR is_table_partitioned('XML_DOCUMENT','DOC_PARTITION_DATE') = false then
    IF f2_flag then
    f2_flag:=false;
        DELETE FROM  XML_DOCUMENT doc 
        WHERE doc.document_id IN (
           SELECT tpic.document_id FROM temp11g_xml_document tpic WHERE  
             (tpic.ecid IS NULL AND mod (tpic.cikey, p_dop)=p_thread) 
          OR (tpic.ecid IS NOT NULL and mod (dbms_utility.get_hash_value(tpic.ecid,0,p_dop), p_dop)=p_thread))
          AND rownum < p_batch_size;
        IF SQL%FOUND THEN
            v_deleted := true;
            f2_flag := true;
            debug_purge('XML_DOCUMENT',null,write_file);
            COMMIT ;
        END IF;
    END IF;
	END IF;

v_stmt := 'Purge DOCUMENT_DLV_MSG_REF' ;
    IF f3_flag then
    f3_flag:=false;
        DELETE FROM DOCUMENT_DLV_MSG_REF d WHERE d.document_id IN ( 
           SELECT tpic.document_id FROM temp11g_document_dlv_msg_ref tpic WHERE  
             (tpic.ecid IS NULL AND mod (tpic.cikey, p_dop)=p_thread) 
          OR (tpic.ecid IS NOT NULL and mod (dbms_utility.get_hash_value(tpic.ecid,0,p_dop), p_dop)=p_thread))
          AND rownum <= p_batch_size;
        IF SQL%FOUND THEN
            v_deleted := true;
            f3_flag := true;
            debug_purge('DOCUMENT_DLV_MSG_REF',null,write_file);
            COMMIT ;
        END IF;
    END IF;

v_stmt := 'Purge DOCUMENT_CI_REF' ;
    IF cursor_flag then
        FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
           DELETE FROM DOCUMENT_CI_REF WHERE cikey = l_cube_instances(i);
        debug_purge('DOCUMENT_CI_REF',null,write_file);
        COMMIT ;
    END IF;

v_stmt := 'Purge DLV_MESSAGE' ;
    IF f4_flag then
    f4_flag:=false;
       DELETE FROM DLV_MESSAGE m where message_guid in (
        select tddmr.message_guid from  temp11g_document_dlv_msg_ref tddmr where  
           mod (dbms_utility.get_hash_value(tddmr.message_guid,0,p_dop), p_dop)=p_thread
          )
          AND rownum <= p_batch_size;
        IF SQL%FOUND THEN
            v_deleted := true;
            f4_flag := true;
            debug_purge('DLV_MESSAGE',null,write_file);
            COMMIT ;
        END IF;
    END IF;

v_stmt := 'Purge DLV_SUBSCRIPTION';
    IF cursor_flag then
        FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
           DELETE FROM DLV_SUBSCRIPTION WHERE cikey = l_cube_instances(i);
        debug_purge('DLV_SUBSCRIPTION',null,write_file);
        COMMIT;
    END IF;

v_stmt := 'Purge DLV_AGGREGATION';
    IF cursor_flag then
        FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
           DELETE FROM DLV_AGGREGATION WHERE cikey = l_cube_instances(i);
        debug_purge('DLV_AGGREGATION',null,write_file);
        COMMIT;
    END IF;

v_stmt := 'Purge VARIABLE_SENSOR_VALUES';
    IF cursor_flag then
        FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
           DELETE FROM VARIABLE_SENSOR_VALUES WHERE PROCESS_INSTANCE_ID = l_cube_instances(i);
        debug_purge('VARIABLE_SENSOR_VALUES',null,write_file);
        COMMIT;
    END IF;

v_stmt := 'Purge FAULT_SENSOR_VALUES';
    IF cursor_flag then
        FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
           DELETE FROM FAULT_SENSOR_VALUES WHERE PROCESS_INSTANCE_ID = l_cube_instances(i);
        debug_purge('FAULT_SENSOR_VALUES',null,write_file);
        COMMIT;
    END IF;

v_stmt := 'Purge ACTIVITY_SENSOR_VALUES';
    IF cursor_flag then
        FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
           DELETE FROM ACTIVITY_SENSOR_VALUES WHERE PROCESS_INSTANCE_ID = l_cube_instances(i);
        debug_purge('ACTIVITY_SENSOR_VALUES',null,write_file);
        COMMIT;
    END IF;

v_stmt := 'Purge CUBE_INSTANCE';
    IF purge_partitioned_component = true OR is_table_partitioned('CUBE_INSTANCE','CPST_INST_CREATED_TIME') = false then
    IF cursor_flag then
        FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
           DELETE FROM CUBE_INSTANCE WHERE cikey = l_cube_instances(i);
        debug_purge('CUBE_INSTANCE',null,write_file);
        commit; 
    END IF;
	END IF;
    
    -- ============= BPMN Tables =====================
    
    v_stmt := 'Purge BPM_AUDIT_QUERY' ;
    IF cursor_flag THEN
        FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
           DELETE FROM BPM_AUDIT_QUERY WHERE component_instance_id = to_char(l_cube_instances(i));
        debug_purge('BPM_AUDIT_QUERY', null, write_file);
        COMMIT;
    END IF;

    v_stmt := 'Purge BPM_MEASUREMENT_ACTIONS' ;
    IF cursor_flag THEN
       FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
           DELETE FROM BPM_MEASUREMENT_ACTIONS WHERE component_instance_id = to_char(l_cube_instances(i));
       debug_purge('BPM_MEASUREMENT_ACTIONS', null, write_file);
       COMMIT;
    END IF;

    v_stmt := 'Purge BPM_MEASUREMENT_ACTION_EXCEPS' ;
    IF cursor_flag THEN
       FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
           DELETE FROM BPM_MEASUREMENT_ACTION_EXCEPS WHERE component_instance_id = to_char(l_cube_instances(i));
       debug_purge('BPM_MEASUREMENT_ACTION_EXCEPS', null, write_file);
       COMMIT;
    END IF;

    v_stmt := 'Purge BPM_CUBE_AUDITINSTANCE' ;
    IF cursor_flag THEN
       FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
           DELETE FROM BPM_CUBE_AUDITINSTANCE WHERE componentinstanceid = to_char(l_cube_instances(i));
       debug_purge('BPM_CUBE_AUDITINSTANCE', null, write_file);
       COMMIT;
    END IF;

    v_stmt := 'Purge BPM_CUBE_TASKPERFORMANCE' ;
    IF cursor_flag THEN
       FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
           DELETE FROM BPM_CUBE_TASKPERFORMANCE WHERE componentinstanceid = to_char(l_cube_instances(i));
        debug_purge('BPM_CUBE_TASKPERFORMANCE', null, write_file);
       COMMIT;
    END IF;

    v_stmt := 'Purge BPM_CUBE_PROCESSPERFORMANCE' ;
    IF cursor_flag THEN
       FORALL i in l_cube_instances.FIRST..l_cube_instances.LAST
           DELETE FROM BPM_CUBE_PROCESSPERFORMANCE WHERE componentinstanceid = to_char(l_cube_instances(i));
       debug_purge('BPM_CUBE_PROCESSPERFORMANCE', null, write_file);
       COMMIT;
    END IF;
    cursor_flag := false;
    if l_num_rows = p_batch_size then
        cursor_flag := true;
        v_deleted := true;
    end if;  
    v_counter := v_counter + 1;

    -- exit loop if out of ime
    IF (sysdate >= v_stoptime) THEN
       v_deleted := FALSE;
       CLOSE c_temp_cube_instance_thread;
       return FALSE ;
    END IF;
END LOOP;
    CLOSE c_temp_cube_instance_thread;
    return TRUE;

EXCEPTION
  when others then
    log_error(v_stmt, write_file);
    CLOSE c_temp_cube_instance_thread;
    raise;
END deleteComponentInstances;

