/*
*
*
* pruneOpenCompositeIDs.sql
*
* Copyright (c) 2010, 2020, Oracle and/or its affiliates. 
*
*    NAME
*      pruneOpenCompositeIDs.sql - BPEL purge prune script 
*
*
*    DESCRIPTION
*     This script is used to prune a list of composite instance ids passed in 
*     an idTable. BPEL uses following criteria to identify a non-purgeable instance :
*     - ci state is <5 (instance is still open)
*     - ci.modify_date is in retention period
*     - any recoverable invoke message exists
*
*     Parameters:
*     p_older_than TIMESTAMP : Retention period for BPEL instances
*     p_id_table : this is the temp table with purge able composite instance ids
*     p_prune_table : Add to this table the open instances
*
*/
PROCEDURE pruneOpenECIDs ( p_id_table    IN VARCHAR2,
                      p_prune_table IN VARCHAR2,
                      p_older_than in TIMESTAMP,
                      write_file in utl_file.file_type default null)
AS
  v_stmt  VARCHAR2(2000);
BEGIN

IF p_prune_table is not null then

    v_stmt := 'INSERT INTO ' || p_prune_table 
            || ' SELECT comp.ECID from CUBE_INSTANCE ci, ' 
            || p_id_table 
            || ' comp WHERE comp.ECID = ci.ECID AND ( ci.STATE < 5 '
            || ' OR ci.MODIFY_DATE >= TO_TIMESTAMP(''RETENTION_PERIOD''))' ;

    v_stmt := REPLACE(v_stmt,'RETENTION_PERIOD',p_older_than);
    EXECUTE immediate v_stmt;
    debug_purge_both(p_prune_table, 'Inserted (bpel) for open ci ',write_file);

    --Recoverable invoke messages cannot be purged.These are in state 0 or 1
    v_stmt := 'INSERT INTO ' || p_prune_table 
           || ' SELECT dlv.ECID from DLV_MESSAGE dlv, ' 
           || p_id_table 
           || ' comp WHERE comp.ECID = dlv.ECID AND dlv.DLV_TYPE=1 AND dlv.STATE in (0,1)' ;
    EXECUTE immediate v_stmt;
    debug_purge_both(p_prune_table, 'Inserted (bpel) for dlv_message '
            ,write_file);
END IF;

/*
v_stmt := 'DELETE FROM ' || p_id_table || ' comp WHERE EXISTS ' ||
              ' (SELECT 1 FROM DLV_MESSAGE dlv WHERE dlv.ECID = comp.ECID AND dlv.DLV_TYPE=1 AND dlv.STATE in (0,1) )';
EXECUTE immediate v_stmt;
debug_purge(p_id_table, 'Deleted (bpel) for open ci');

v_stmt := 'DELETE FROM ' || p_id_table || ' comp WHERE EXISTS ' ||
              ' (SELECT 1 FROM CUBE_INSTANCE ci WHERE ci.ECID = comp.ECID ' ||
              ' AND (ci.STATE < 5 or ci.MODIFY_DATE >= TO_TIMESTAMP(''RETENTION_PERIOD''))  )';

v_stmt := REPLACE(v_stmt,'RETENTION_PERIOD',p_older_than);
*/
v_stmt := 'DELETE FROM ' || p_id_table || ' comp WHERE EXISTS ' 
       || ' (SELECT 1 FROM ' || p_prune_table 
       || ' prune WHERE prune.ECID = comp.ECID )' ;

EXECUTE immediate v_stmt;
debug_purge_both(p_id_table, 'Deleted (bpel) for dlv_message or open ci ',write_file);

COMMIT;

EXCEPTION
  when others then
    log_error_both(v_stmt,write_file);
    raise;

END pruneOpenECIDs;
