/*
*
* deleteComponentInstances.sql
*
* Copyright (c) 2010, 2023, Oracle and/or its affiliates. 
*
*    NAME
*      deleteComponentInstances.sql - BPEL purge script 
*
*
*    DESCRIPTION
*     This script is used to purge BPEL tables provided with a list of composite instance ids
*     in an idTable. 
*
*    For a user with multi cpu machine, usage of parallel dml may help. Add this to master script.
*      execute immediate 'alter session enable parallel dml';
*
*/

PROCEDURE truncate_temp_tables
AS
BEGIN
  --before starting clean up temp tables
  EXECUTE IMMEDIATE 'TRUNCATE TABLE temp_cube_instance';
  EXECUTE IMMEDIATE 'TRUNCATE TABLE temp_document_ci_ref';
  EXECUTE IMMEDIATE 'TRUNCATE TABLE temp_document_dlv_msg_ref';
  EXECUTE IMMEDIATE 'TRUNCATE TABLE temp_xml_document';
END truncate_temp_tables;


/*
* Procedure: purge_bpel_tables
*
* Description
* This procedure purges following BPEL tables based on temp tables created in earlier step.
*
*/
PROCEDURE purge_bpel_tables
                 (p_id_table   IN VARCHAR2, purge_partitioned_component in boolean default false,
                  write_file in utl_file.file_type default null)
AS
  v_stmt  VARCHAR2(2000);
BEGIN
--Create additional temp tables
v_stmt := 'Create temp_document_ci_ref' ;
    INSERT INTO temp_document_ci_ref (document_id) SELECT document_id
        FROM document_ci_ref d, temp_cube_instance t WHERE d.cikey = t.cikey;
    debug_purge_both('temp_document_ci_ref', 'Inserted = ',write_file);

v_stmt :=  'INSERT INTO temp_document_dlv_msg_ref (message_guid,document_id) '||
           'SELECT DISTINCT m.message_guid,ddmr.document_id '||
           'FROM document_dlv_msg_ref ddmr, dlv_message m ' ||
           'WHERE ddmr.message_guid = m.message_guid ' ||
           'AND EXISTS (SELECT 1 FROM ' || p_id_table || ' comp '|| ' where m.flow_id = comp.flow_id)' ;

    EXECUTE immediate v_stmt;
    debug_purge_both('temp_document_dlv_msg_ref', 'Inserted = ',write_file);

-- Being purge of BPEL data

v_stmt := 'Purge HEADERS_PROPERTIES' ;
      DELETE FROM HEADERS_PROPERTIES WHERE message_guid IN (SELECT message_guid FROM temp_document_dlv_msg_ref) ;
     debug_purge_both('HEADERS_PROPERTIES','Deleted = ',write_file);
     COMMIT;

v_stmt := 'Purge AG_INSTANCE' ;
    DELETE FROM AG_INSTANCE WHERE cikey in (SELECT tpic.cikey FROM temp_cube_instance tpic);
    debug_purge_both('AG_INSTANCE','Deleted = ',write_file);
    COMMIT;

v_stmt := 'Purge TEST_DETAILS' ;
    DELETE FROM TEST_DETAILS WHERE cikey in (SELECT tpic.cikey FROM temp_cube_instance tpic);
    debug_purge_both('TEST_DETAILS','Deleted = ',write_file);
    COMMIT;
    
v_stmt := 'Purge CUBE_SCOPE' ;
   if purge_partitioned_component = true OR is_table_partitioned('CUBE_SCOPE','CI_PARTITION_DATE') = FALSE then
		DELETE FROM CUBE_SCOPE WHERE cikey in (SELECT tpic.cikey FROM temp_cube_instance tpic);
		debug_purge_both('CUBE_SCOPE','Deleted = ',write_file);
		COMMIT;
	end if;	

v_stmt := 'Purge CI_INDEXES' ;
    DELETE FROM CI_INDEXES WHERE cikey in (SELECT tpic.cikey FROM temp_cube_instance tpic);
    debug_purge_both('CI_INDEXES','Deleted = ',write_file);
    COMMIT;

v_stmt := 'Purge WORK_ITEM' ;
    DELETE FROM WORK_ITEM WHERE cikey in (SELECT tpic.cikey FROM temp_cube_instance tpic);
    debug_purge_both('WORK_ITEM','Deleted = ',write_file);
    COMMIT ;

v_stmt := 'Purge WI_FAULT' ;
    DELETE FROM WI_FAULT WHERE cikey in (SELECT tpic.cikey FROM temp_cube_instance tpic);
    debug_purge_both('WI_FAULT','Deleted = ',write_file);
    COMMIT ;

v_stmt := 'Purge XML_DOCUMENT';
    -- TODO optimize the if condition . Should not call is_table_partitioned for each loop)  
    if purge_partitioned_component = true OR is_table_partitioned('XML_DOCUMENT','DOC_PARTITION_DATE') = FALSE then
    DELETE FROM  XML_DOCUMENT d WHERE document_id in
              ( SELECT t1.document_id FROM  temp_document_dlv_msg_ref t1
                union  
                SELECT t2.document_id FROM temp_document_ci_ref t2
              );
    debug_purge_both('XML_DOCUMENT','Deleted = ',write_file);
    COMMIT ;
	end if;
    
v_stmt := 'Purge DOCUMENT_DLV_MSG_REF' ;
    DELETE FROM DOCUMENT_DLV_MSG_REF  WHERE document_id in (SELECT document_id FROM temp_document_dlv_msg_ref t);
    debug_purge_both('DOCUMENT_DLV_MSG_REF','Deleted = ',write_file);
    COMMIT ;

v_stmt := 'Purge DOCUMENT_CI_REF' ;
    DELETE FROM DOCUMENT_CI_REF WHERE cikey in (SELECT tpic.cikey FROM temp_cube_instance tpic);
    debug_purge_both('DOCUMENT_CI_REF','Deleted = ',write_file);
    COMMIT ;

v_stmt := 'Purge DLV_MESSAGE' ;
    DELETE FROM DLV_MESSAGE  where message_guid in ( SELECT message_guid from temp_document_dlv_msg_ref );
    debug_purge_both('DLV_MESSAGE','Deleted = ',write_file);
    COMMIT ;

v_stmt := 'Purge DLV_SUBSCRIPTION';
    DELETE FROM DLV_SUBSCRIPTION s WHERE cikey in (SELECT tpic.cikey FROM temp_cube_instance tpic);
    debug_purge_both('DLV_SUBSCRIPTION','Deleted = ',write_file);
    COMMIT ;

v_stmt := 'Purge DLV_AGGREGATION';
    DELETE FROM DLV_AGGREGATION WHERE cikey in (SELECT tpic.cikey FROM temp_cube_instance tpic);
    debug_purge_both('DLV_AGGREGATION','Deleted = ',write_file);
    COMMIT ;

if purge_partitioned_component = true
OR is_table_partitioned('VARIABLE_SENSOR_VALUES','CREATION_DATE') = FALSE then
   v_stmt := 'Purge VARIABLE_SENSOR_VALUES';
       DELETE FROM VARIABLE_SENSOR_VALUES WHERE PROCESS_INSTANCE_ID in (SELECT tpic.cikey FROM temp_cube_instance tpic);
       debug_purge('VARIABLE_SENSOR_VALUES');
       COMMIT ;
end if;

if purge_partitioned_component = true
OR is_table_partitioned('FAULT_SENSOR_VALUES','CREATION_DATE') = FALSE then
   v_stmt := 'Purge FAULT_SENSOR_VALUES';
       DELETE FROM FAULT_SENSOR_VALUES WHERE PROCESS_INSTANCE_ID in (SELECT tpic.cikey FROM temp_cube_instance tpic);
       debug_purge('FAULT_SENSOR_VALUES');
       COMMIT ;
end if;

if purge_partitioned_component = true
OR is_table_partitioned('ACTIVITY_SENSOR_VALUES','CREATION_DATE') = FALSE then
   v_stmt := 'Purge ACTIVITY_SENSOR_VALUES';
       DELETE FROM ACTIVITY_SENSOR_VALUES WHERE PROCESS_INSTANCE_ID in (SELECT tpic.cikey FROM temp_cube_instance tpic);
       debug_purge('ACTIVITY_SENSOR_VALUES');
       COMMIT ;
end if;

v_stmt := 'Purge CUBE_INSTANCE';
    IF purge_partitioned_component = true OR is_table_partitioned('CUBE_INSTANCE','CPST_INST_CREATED_TIME') = false then
    -- TODO optimize the if condition . Should not call is_table_partitioned for each loop)   
    DELETE FROM cube_instance ci WHERE cikey in (SELECT tpic.cikey FROM temp_cube_instance tpic);
    debug_purge_both('CUBE_INSTANCE','Deleted = ',write_file);
    COMMIT ;
    END IF;
    
EXCEPTION
  when others then
    log_error_both(v_stmt,write_file);
    raise;

END purge_bpel_tables;
/*
* Procedure : deleteComponentInstances
*
* Description: 
*  This procedure is provided a table with composite instance ids that are purge able.
*  Following temp tables are created and passed to purge_bpel_tables procedure.
*
*  1. temp_cube_instance
*  2. temp_document_ci_ref
*  3. temp_document_dlv_msg_ref
*
*
*  Parameters:
*  idTable : Name of the id table that contains list of purge able composite instance IDs
*  
*/

PROCEDURE deleteComponentInstances(p_id_table   IN VARCHAR2, 
                    purge_partitioned_component in boolean default false,
                    write_file in utl_file.file_type default null)
AS
  v_stmt  VARCHAR2(2000);
BEGIN

    truncate_temp_tables();

v_stmt := 'INSERT INTO temp_cube_instance ' ||
          'SELECT DISTINCT cikey, ci.flow_id FROM cube_instance ci, '
          || p_id_table || ' comp WHERE ci.flow_id = comp.flow_id';
EXECUTE immediate v_stmt;

    debug_purge_both('temp_cube_instance', 'Inserted = ',write_file);

    purge_bpel_tables(p_id_table,purge_partitioned_component,write_file);

EXCEPTION
  when others then
    log_error_both(v_stmt,write_file);
    raise;

END deleteComponentInstances;

FUNCTION abortComponentInstances(min_creation_date IN TIMESTAMP, max_creation_date IN TIMESTAMP, lookupcompquery IN VARCHAR2 , batch_size IN INTEGER, cmpst_name IN VARCHAR2,max_runtime IN INTEGER, update_retention IN BOOLEAN ) RETURN INTEGER AS
	batch_total_count   INTEGER := 0;
	common_abort_query VARCHAR2(4000);
	filter_instances_query VARCHAR2(10000) := 'SELECT ci.cikey FROM CUBE_INSTANCE ci, SCA_FLOW_INSTANCE cmpt WHERE ci.flow_id=cmpt.flow_id AND ci.state not in(8,5,9,4) ';
BEGIN
 --bpel started flow
            IF lookupcompquery IS NOT NULL THEN
				filter_instances_query := filter_instances_query || 'AND' || lookupcompquery;
			END If;
			
			IF cmpst_name IS NOT NULL THEN
				filter_instances_query := filter_instances_query || ' AND ci.COMPOSITE_NAME =' || '''' || cmpst_name || ''''; 
			END IF;
			
        --dlv_message
            common_abort_query := 'UPDATE dlv_message  SET  state = 3 ,EXT_STRING1=''PLSQL ABORT'' WHERE  state IN ( 0,1,4)';
            common_abort_query := common_abort_query  || ' AND cikey IN(' || filter_instances_query || ')' ;
			batch_total_count := batch_update_func(common_abort_query, batch_size, max_runtime);
			COMMIT;
			dbms_output.put_line('Number of dlv_messages entries marked aborted with main criteria ::::' || batch_total_count);

			common_abort_query := 'UPDATE dlv_message  SET  state = 3 WHERE  state IN ( 0,1,4)';
			    
			IF cmpst_name IS NOT NULL THEN
				common_abort_query := common_abort_query || ' AND COMPOSITE_NAME =' || '''' || cmpst_name || ''''; 
			ELSE  
			    common_abort_query := common_abort_query || ' AND COMPOSITE_NAME IN (SELECT DISTINCT  ci.COMPOSITE_NAME FROM CUBE_INSTANCE ci, 					SCA_FLOW_INSTANCE cmpt WHERE ci.flow_id=cmpt.flow_id AND ci.state not in(8,5,9,4) AND ' || lookupcompquery || ')'; 
			END IF;	
			
			IF min_creation_date IS NOT NULL THEN
				common_abort_query := common_abort_query || ' AND RECEIVE_DATE >=' || '''' || min_creation_date || '''';			
			END IF;	
			IF max_creation_date IS NOT NULL THEN
				common_abort_query := common_abort_query || ' AND RECEIVE_DATE <=' || '''' || max_creation_date || '''';			
			END IF;	
                        batch_total_count := batch_update_func(common_abort_query, batch_size, max_runtime);
			COMMIT;
			dbms_output.put_line('Number of dlv_messages entries marked aborted with composite name= ' || batch_total_count);
			
            

		-- work_Item
			common_abort_query := 'UPDATE work_item  SET  state = 10  WHERE  state IN (1,2,3,5,13)';
            common_abort_query := common_abort_query  || ' AND cikey IN(' || filter_instances_query || ')' ;
			batch_total_count := batch_update_func(common_abort_query, batch_size, max_runtime);
			COMMIT;
			dbms_output.put_line('Number of work_item entries marked aborted = ' || batch_total_count);

		--dlv_subscription
			common_abort_query := 'UPDATE dlv_subscription  SET  state = -1  WHERE  state IN ( 0,1,4)';
            common_abort_query := common_abort_query  || ' AND cikey IN(' || filter_instances_query || ')' ;
			batch_total_count := batch_update_func(common_abort_query, batch_size, max_runtime);
			COMMIT;
			dbms_output.put_line('Number of dlv_subscription entries marked aborted = ' || batch_total_count);

 		-- cube_instance
			common_abort_query := 'UPDATE CUBE_INSTANCE  SET  state = 8 ,EXT_STRING1=''PLSQL ABORT''';
			IF update_retention THEN
				common_abort_query := common_abort_query || ', MODIFY_DATE=sysdate ';
			END IF;			common_abort_query := common_abort_query || '  WHERE   state not in(8,5,9,4)';
            common_abort_query := common_abort_query || ' AND cikey IN(' || filter_instances_query || ')' ;
			batch_total_count := batch_update_func(common_abort_query, batch_size, max_runtime);
			COMMIT;
			dbms_output.put_line('Number of CUBE_INSTANCE entries marked aborted = ' || batch_total_count);
--  bpel finished
    RETURN batch_total_count;
EXCEPTION
    WHEN OTHERS THEN
        RETURN -1;
END abortComponentInstances;

