/*
Rem
Rem Copyright (c) 2006, 2024, Oracle and/or its affiliates. 
Rem All rights reserved.
Rem
Rem    NAME
Rem      deleteInstances.sql <one-line expansion of the name>
Rem
Rem    DESCRIPTION
Rem      SOA 12c Purge
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem       apfwkr   11/27/24 - CI #37331393 of
Rem       apfwkr   11/27/24 - apfwkr_blr_backport_37249329_12.2.1.4.240919soabp
Rem                           from st_pcbpel_12.2.1.4.0soabp
Rem       apfwkr   11/13/24 - BLR# 37278649 of jbahadur_bug-37249329_oneoff
Rem       apfwkr   11/13/24 - from st_pcbpel_12.2.1.4.0soabp.
Rem       apfwkr   11/13/24 - BLR# 37278649
Rem                           apfwkr_blr_backport_37249329_12.2.1.4.240919soabp
Rem       apfwkr   02/22/24 - CI# 36329882 of
Rem       apfwkr   02/22/24 - apfwkr_blr_backport_28642751_12.2.1.4.231122soabp
Rem                           from st_pcbpel_12.2.1.4.0soabp
Rem       apfwkr   02/12/24 - BLR# 36283216 of wfchen_bug-28642751_12215 .
Rem       apfwkr   02/12/24 - BLR# 36283216
Rem       apfwkr   02/12/24 - apfwkr_blr_backport_28642751_12.2.1.4.231122soabp
Rem                           from main
Rem       apfwkr   01/19/23 - CI# 34996638 of
Rem       apfwkr   01/19/23 - apfwkr_blr_backport_33323056_12.2.1.4.220827soabp
Rem       apfwkr   01/19/23 - from st_pcbpel_12.2.1.4.0soabp.
Rem       apfwkr   01/19/23 - CI# 34996638
Rem       apfwkr   01/19/23 - apfwkr_ci_backport_33323056_12.2.1.4.0soabp
Rem       apfwkr   12/08/22 - BLR# 34874653 of jbahadur_bug-33323056_1221 .
Rem       apfwkr   12/08/22 - BLR# 34874653
Rem       apfwkr   12/08/22 - apfwkr_blr_backport_33323056_12.2.1.4.220827soabp
Rem       apfwkr   12/08/22 - from main
Rem       apfwkr   11/04/20 - Backport
Rem                           apfwkr_blr_backport_31572611_12.2.1.4.200917soabp
Rem                           from st_pcbpel_12.2.1.4.0soabp
Rem       apfwkr   10/19/20 - Backport apfwkr_blr_backport_31572611_12.2.1.4.0
Rem                           from main
Rem       apfwkr   07/23/20 - Backport
Rem                           apfwkr_blr_backport_31589423_12.2.1.4.200524soabp
Rem                           from st_pcbpel_12.2.1.4.0soabp
Rem       apfwkr   07/20/20 - Backport apfwkr_blr_backport_31589423_12.2.1.4.0
Rem                           from main
Rem       apfwkr   04/20/20 - Backport
Rem                           shabdull_bug-31199221_r2_upgradescriptissue_from_30102671
Rem                           from st_pcbpel_12.2.1.4.0soabp
Rem       apfwkr   07/16/20 - Backport linlxu_bug-31589423_apps_1221 from
Rem                           st_pcbpel_12.2.1.0.0
Rem       shabdull 12/17/19 - Backport apfwkr_blr_backport_30374610_12.2.1.1.0
Rem                           from st_pcbpel_12.2.1.1.0
Rem       apfwkr   11/20/19 - Backport linlxu_bug-30374610_12212-main from main
Rem       apfwkr   10/28/19 - Backport linlxu_bug-30374610_12212-main from main
Rem       apfwkr   10/16/20 - Backport apfwkr_blr_backport_31572611_12.2.1.3.0
Rem                           from main
Rem       apfwkr   10/15/20 - Backport
Rem                           shabdull_blr_backport_31572611_12.2.1.2.0 from
Rem                           st_pcbpel_12.2.1.2.0
Rem       shabdull 07/16/20 - Backport linlxu_bug-31572611_12212 from
Rem                           st_pcbpel_12.2.1.2.0
Rem       mbousamr 11/15/12 - Modified for SOA 12c.
Rem       sanjain  04/29/10 - Enabling Mediator purge
Rem
*/
    function  getComponentPartitionInfo return component_partition_info is
     componentPartInfo component_partition_info ;
    begin
        log_info('checking for  partitions');
        componentPartInfo := component_partition_info(); 
        componentPartInfo.fabricPartitioned := soa_fabric.getComponentPartitionInfo;
        componentPartInfo.bpelPartitioned := soa_orabpel.getComponentPartitionInfo;
	componentPartInfo.mediatorPartitioned := soa_mediator.getComponentPartitionInfo;
        if soa_b2b.isComponentPartitioned then
           componentPartInfo.b2bPartitioned := 'Y';
        else
           componentPartInfo.b2bPartitioned := 'N';
        end if;
        if soa_workflow.isComponentPartitioned then
           componentPartInfo.workflowPartitioned := 'Y';
        else
           componentPartInfo.workflowPartitioned := 'N';
        end if;
        if soa_decision.isComponentPartitioned then
           componentPartInfo.decisionPartitioned := 'Y';
        else
           componentPartInfo.decisionPartitioned := 'N';
        end if;
        log_info('done checking for partitions');
        return componentPartInfo;
    end;
 
  function make_composite_dn(soa_partition_name in varchar2, composite_name in varchar2, composite_revision in varchar2) return varchar2 is

     composite_dn         varchar2(500) default null;
     composite_name_x     varchar2(500) default null;
     composite_revision_x varchar2(50)  default null;
     soa_partition_name_x varchar2(200) default null;
  
    begin
        if composite_name is not null then
          composite_name_x := ' and e.composite = ''' 
                                || composite_name ||''' ';
        end if; 
        
        if composite_name is not null
        and composite_revision is not null then
          composite_revision_x := ' and e.revision = ''' 
                                    || composite_revision ||''' ';
        end if;
        
        if composite_name is not null then
           if soa_partition_name is not null then
            soa_partition_name_x := ' and e.sca_partition_id = p.id  and p.name = ''' || soa_partition_name ||''' ';
           else 
            soa_partition_name_x := ' and e.sca_partition_id = p.id  and p.name = ''default'' ';
           end if; 
        end if;
        
        composite_dn := composite_name_x 
                     || composite_revision_x 
                     || soa_partition_name_x;
        
        log_info('composite_dn = ' || composite_dn);

        return composite_dn;    
	    
    end make_composite_dn;
      
    PROCEDURE dangling_delete(retention_period in date default null, p_transaction_size in NUMBER, v_stop_time in date, write_file in utl_file.file_type default null, purge_partitioned_component in boolean default false) is 
     p_retention_period date := retention_period;
    begin
       IF (p_retention_period IS NULL) THEN
          p_retention_period := sysdate -7;
       END IF;

      IF (sysdate >= v_stop_time) THEN
           return;
      END IF;

      soa_orabpel.dangling_delete(p_retention_period, p_transaction_size, v_stop_time, write_file, purge_partitioned_component);
      IF (sysdate >= v_stop_time) THEN
           return;
      END IF;
      soa_mediator.dangling_delete(p_retention_period, p_transaction_size, v_stop_time, write_file, purge_partitioned_component);
      IF (sysdate >= v_stop_time) THEN
           return;
      END IF;
      soa_workflow.dangling_delete(p_retention_period, p_transaction_size, v_stop_time, write_file, purge_partitioned_component);
      IF (sysdate >= v_stop_time) THEN
           return;
      END IF;
      soa_fabric.dangling_delete(p_retention_period, p_transaction_size, v_stop_time, write_file, purge_partitioned_component);
      IF (sysdate >= v_stop_time) THEN
           return;
      END IF;
    end dangling_delete;

   /**
     * We are seeing instance tracking bugs, where flow_instance.active_component_count is wrongly set to zero with open BPEL/BPMN instances.
     * So adding defensive check to avoid purge of live data
     */    
    procedure pruneOpenFlowIDs (purge_id_table in varchar2, prune_running_insts_table in varchar2, write_file in utl_file.file_type default null) is 
    begin
       log_info_both(' pruneOpenFlowIDs  started ');
       soa_orabpel.pruneOpenFlowIDs (purge_id_table, prune_running_insts_table, write_file);
       soa_mediator.pruneOpenFlowIDs (purge_id_table, prune_running_insts_table, write_file);

       /* For now we are only doing this for BPEL/BPM instances
       soa_workflow.pruneOpenFlowIDs (purge_id_table, prune_running_insts_table, write_file);
       soa_decision.pruneOpenFlowIDs (purge_id_table, prune_running_insts_table, write_file);
        */
        log_info_both(' pruneOpenFlowIDs  Complete ');       
    end pruneOpenFlowIDs;

   /**
     * procedure delete_flow_instances
     *
     * Deletes the flow instances and all the associated rows in other fabric tables that
     * reference this flow instance directly or indirectly. 
     */    
    procedure delete_flow_instances(instance_id_table in varchar2,
                             purge_partitioned_component in boolean,
                             componentPartInfo in component_partition_info,
                             stoptime in date,
                             max_count in integer,
                             min_creation_date in timestamp,
                             max_creation_date in timestamp,
                             retention_period in timestamp,
                             write_file in utl_file.file_type default null
                                        )  is   
  

   
    begin
       IF (sysdate >= stoptime) THEN
          return;
       END IF;
       if purge_partitioned_component = true 
        OR componentPartInfo.bpelPartitioned='N' 
        OR componentPartInfo.bpelPartitioned='P'then
           log_info_both('calling soa_orabpel.deleteComponentInstances'
                         ,write_file);
           soa_orabpel.deleteComponentInstances
               (instance_id_table,purge_partitioned_component,write_file);
           log_info_both('completed soa_orabpel.deleteComponentInstances'
                         ,write_file);
       end if; 

       IF (sysdate >= stoptime) THEN
           return;
       END IF;
       if purge_partitioned_component = true 
        OR componentPartInfo.workflowPartitioned='N' then   
           log_info_both('calling workflow.deleteComponentInstances'
                         ,write_file);
           soa_workflow.deleteComponentInstances(instance_id_table,write_file);
           log_info_both('completed workflow.deleteComponentInstances'
                         ,write_file);
        end if;      

       IF (sysdate >= stoptime) THEN
           return;
       END IF;
       if purge_partitioned_component = true 
        OR componentPartInfo.mediatorPartitioned='N' 
        OR componentPartInfo.mediatorPartitioned='P' then   
           log_info_both('calling mediator.deleteComponentInstances'
                         ,write_file);
           soa_mediator.deleteComponentInstances
                 (instance_id_table,
                  purge_partitioned_component,
                  max_count,
                  min_creation_date,
                  max_creation_date,
                  retention_period,
                  write_file);
           log_info_both('completed mediator.deleteComponentInstances'
                         ,write_file);
       end if;

       IF (sysdate >= stoptime) THEN
           return;
       END IF;
       if purge_partitioned_component = true 
        OR componentPartInfo.decisionPartitioned='N' then   
           log_info_both('calling decision.deleteComponentInstances'
                         ,write_file);
           soa_decision.deleteComponentInstances(instance_id_table,write_file);
           log_info_both('completed decision.deleteComponentInstances'
                         ,write_file);
       end if;

       IF (sysdate >= stoptime) THEN
           return;
       END IF;
       if purge_partitioned_component = true 
        OR componentPartInfo.fabricPartitioned='N' 
        OR componentPartInfo.fabricPartitioned='P' then   
           log_info_both('calling fabric.deleteComponentInstances'
                         ,write_file);
           soa_fabric.deleteComponentInstances
               (instance_id_table,purge_partitioned_component,write_file);
           log_info_both('completed fabric.deleteComponentInstances'
                         ,write_file);
       end if; 

      
    exception
      when others then
        log_error_both('ERROR (delete_flow_instances)',write_file);
        rollback;
        raise;
    end delete_flow_instances;

   /**
     * procedure delete_stat_alert_tables
     *
     */
    procedure delete_stat_alert_tables(stoptime in date,
                             max_count in integer,
                             batch_size in integer,
                             retention_period in timestamp,
                             write_file in utl_file.file_type default null
                                        )  is


    v_stmt  VARCHAR2(500);
    v_deleted boolean:=true;
    f1_flag boolean:=true;
    f2_flag boolean:=true;
    f3_flag boolean:=true;
    f4_flag boolean:=true;
    f5_flag boolean:=true;
    f6_flag boolean:=true;
    f7_flag boolean:=true;
    max_count_2 integer := max_count;
    
    begin
           

    WHILE  v_deleted  LOOP

      v_deleted := FALSE;

      IF (sysdate >= stoptime) THEN
         EXIT;
      END IF;
      If f1_flag then
         f1_flag := false;       
         v_stmt := 'DELETE from EIS_CONNECTION_DOWN_TIME '
           || ' where rownum <= ' ||  batch_size;
         IF retention_period is not null then
            v_stmt := v_stmt
              || ' and creation_time <= ' || '''' || retention_period || '''';
         END IF;
         EXECUTE immediate v_stmt;
         IF SQL%FOUND THEN
            v_deleted := true;
            f1_flag := true;
            debug_purge_both('EIS_CONNECTION_DOWN_TIME','Deleted = ',write_file);
         END IF;
         COMMIT;
      END IF;

      IF (sysdate >= stoptime) THEN
         EXIT;
      END IF;
      If f2_flag then
         f2_flag := false;
         v_stmt := 'DELETE from MESSAGE_STATISTICS '
           || ' where rownum <= ' || batch_size;
         IF retention_period is not null then
            v_stmt := v_stmt
              || ' and creation_time <= ' || '''' || retention_period || '''';
         END IF;
         EXECUTE immediate v_stmt;
         IF SQL%FOUND THEN
            v_deleted := true;
            f2_flag := true;
            debug_purge_both('MESSAGE_STATISTICS','Deleted = ',write_file);
         END IF;
         COMMIT;
      END IF;

      IF (sysdate >= stoptime) THEN
         EXIT;
      END IF;
      If f3_flag then
         f3_flag := false;
         v_stmt := 'DELETE from FAULT_ALERT '
           || ' where rownum <= ' || batch_size;
         IF retention_period is not null then
            v_stmt := v_stmt
              || ' and creation_date <= ' || '''' || retention_period || '''';
         END IF;
         EXECUTE immediate v_stmt;
         IF SQL%FOUND THEN
            v_deleted := true;
            f3_flag := true;
            debug_purge_both('FAULT_ALERT','Deleted = ',write_file);
         END IF;
         COMMIT ;
      END IF;

      IF (sysdate >= stoptime) THEN
         EXIT;
      END IF;
      If f4_flag then
         f4_flag := false;         
         v_stmt := 'DELETE from SOA_PURGE_HISTORY '
           || ' where rownum <= ' || batch_size;
         IF retention_period is not null then
            v_stmt := v_stmt
              || ' and start_time <= ' || '''' || retention_period || '''';
         END IF;
         EXECUTE immediate v_stmt;
         IF SQL%FOUND THEN
            v_deleted := true;
            f4_flag := true;
            debug_purge_both('SOA_PURGE_HISTORY','Deleted = ',write_file);
         END IF;
         COMMIT ;
      END IF;

      IF (sysdate >= stoptime) THEN
         EXIT;
      END IF;
      If f5_flag then
         f5_flag := false;
         v_stmt := 'DELETE from BPELNotification '
           || ' where status IN (''SENT'', ''ERROR'') ';
         IF retention_period is not null then
            v_stmt := v_stmt
              || ' and to_timestamp(to_char(CREATEDTIME, ''DD-MON-YYYY HH24:MI:SS''), ''DD-MON-YYYY HH24:MI:SS'') <= ' || '''' || retention_period || '''';
         END IF;
         v_stmt := v_stmt
              || ' and rownum <= ' || batch_size;
         EXECUTE immediate v_stmt;
         IF SQL%FOUND THEN
            v_deleted := true;
            f5_flag := true;
            debug_purge_both('BPELNotification','Deleted = ',write_file);
         END IF;
         COMMIT ;
      END IF;

      IF (sysdate >= stoptime) THEN
         EXIT;
      END IF;
      If f6_flag then
         f6_flag := false;
         v_stmt := 'DELETE from BPELNotification_filtermsgs '
           || ' where rownum <= ' || batch_size;
         IF retention_period is not null then
            v_stmt := v_stmt
              || ' and to_timestamp(to_char(RECEIVED_DATE, ''DD-MON-YYYY HH24:MI:SS''), ''DD-MON-YYYY HH24:MI:SS'') <= ' || '''' || retention_period || '''';
         END IF;
         EXECUTE immediate v_stmt;
         IF SQL%FOUND THEN
            v_deleted := true;
            f6_flag := true;
            debug_purge_both('BPELNotification_filtermsgs','Deleted = ',write_file);
         END IF;
         COMMIT ;
      END IF;

      IF (sysdate >= stoptime) THEN
         EXIT;
      END IF;
      If f7_flag then
         f7_flag := false;
         v_stmt := 'DELETE from BPELNotification_invaladdrs '
           || ' where rownum <= ' || batch_size;
         IF retention_period is not null then
            v_stmt := v_stmt
              || ' and to_timestamp(to_char(INVALIDATED_TIME, ''DD-MON-YYYY HH24:MI:SS''), ''DD-MON-YYYY HH24:MI:SS'') <= ' || '''' || retention_period || '''';
         END IF;
         EXECUTE immediate v_stmt;
         IF SQL%FOUND THEN
            v_deleted := true;
            f7_flag := true;
            debug_purge_both('BPELNotification_invaladdrs','Deleted = ',write_file);
         END IF;
         COMMIT ;
      END IF;

      IF (sysdate >= stoptime) THEN
         EXIT;
      END IF;

      max_count_2 := max_count_2 - batch_size;
      IF max_count_2 <= 0 THEN
         v_deleted := FALSE;
      END If;

    END LOOP;

    end delete_stat_alert_tables;

   /**
     * procedure check_purge_session
     *   If another database session is found the purge will exit.
     *   The SID and SERIAL# can be used to kill the session
     *    if appropriate.
     */
   
    procedure check_purge_session(write_file in utl_file.file_type default null)
    is

    session_found boolean;

    CURSOR c_gvsession is
    select inst_id, sid, serial#
     from gv$session
    where action = 'SOAPURGEEXECUTING'
      and schemaname in
      (SELECT sys_context('USERENV', 'CURRENT_SCHEMA') FROM DUAL); 

    begin

       session_found := FALSE;

       FOR gvsess_rec in c_gvsession
       LOOP
          log_info_both
          ('ERROR: Another SOA Purge(parallel/single)is in progress.',write_file);
          log_info_both('Database gv$session:',write_file);
          log_info_both('  INST_ID  :' || gvsess_rec.inst_id,write_file);
          log_info_both('  SID      :' || gvsess_rec.sid,write_file);
          log_info_both('  SERIAL#  :' || gvsess_rec.serial#,write_file);
          session_found := TRUE;
       END LOOP;

       -- EXIT if another SOA purge (single/parallel) is executing.
       IF session_found THEN
          RAISE_APPLICATION_ERROR
           (-20099, 'ERROR Another SOA Purge(parallel/single)is in progress');
       END IF;

    end check_purge_session;

   /**
     * procedure check_upgrade_session
     *   If the 11g to 12c upgrade is found the purge will exit.
     *   The SID and SERIAL# can be used to kill the session
     *    if appropriate.
     */

    procedure check_upgrade_session(write_file in utl_file.file_type default null)
    is

    session_found boolean;

    CURSOR c_gvsession is
    select inst_id, sid, serial#
     from gv$session
    where action = 'SOAUPGRADEEXECUTING'
      and schemaname in
      (SELECT sys_context('USERENV', 'CURRENT_SCHEMA') FROM DUAL);

    begin

       session_found := FALSE;

       FOR gvsess_rec in c_gvsession
       LOOP
          log_info_both
          ('ERROR: The 11g to 12c Upgrade is in progress.',write_file);
          log_info_both('Database gv$session:');
          log_info_both('  INST_ID  :' || gvsess_rec.inst_id,write_file);
          log_info_both('  SID      :' || gvsess_rec.sid,write_file);
          log_info_both('  SERIAL#  :' || gvsess_rec.serial#,write_file);
          session_found := TRUE;
       END LOOP;

       -- EXIT if the 11g to 12c upgrade is executing.
       IF session_found THEN
          RAISE_APPLICATION_ERROR
           (-20099, 'ERROR The 11g to 12c Upgrade is in progress');
       END IF;

    end check_upgrade_session;

procedure delete_soa_instances ( min_creation_date in timestamp,
                   max_creation_date in timestamp,
                   batch_size in integer default 20000, 
                   -- max_runtime in integer default 60, 
                   stoptime in date default null,
                   retention_period in timestamp default null,
                   purge_partitioned_component in boolean default false,
                   composite_name in  varchar2 default null,
                   composite_revision in varchar2 default null,
                   soa_partition_name in varchar2 default null,
                   ignore_state in boolean default false,
                   PQS in integer default 1, 
                   sql_trace IN boolean default false,
                   inmemory in boolean default false,
		   compbased in boolean default false
                   ) is 
    
    flow_id_sql varchar2(4000);
    v_stmt varchar2(500);
    purge_id_table varchar2(100) := 'temp_flowid_purge';
    -- stoptime date := sysdate + NVL(max_runtime,24*60)/(24*60);
    total_rows integer;
    total_rows_after_pruning integer; 
    purge_loop BOOLEAN:=TRUE;
    purge_stat BOOLEAN:=TRUE;
    iterationCount integer :=1;
    more_rows_to_delete boolean := false;
    prune_running_insts_table varchar2(100) :='temp_prune_running_insts';
    v_retention_period timestamp;
    componentPartInfo component_partition_info;
    composite_dn varchar2(500);
    soa_purge_job_no number;
    v_batch_size number := batch_size;
    -- v_max_runtime number := max_runtime;
    v_PQS number := PQS;
    write_file  utl_file.file_type;

    begin

      write_file := get_file_main;

      -- Check and exit if another purge is currently running.
      check_purge_session(write_file);
      -- Check and exit if the 11g to 12c upgrade is running.
      check_upgrade_session(write_file);
      -- Set ACTION_NAME on SESSION.
      DBMS_APPLICATION_INFO.SET_MODULE(
            MODULE_NAME => 'SINGLEPURGE',
            ACTION_NAME => 'SOAPURGEEXECUTING');

      log_info_both('***************************************',write_file);
      log_info_both('***************************************',write_file);
      log_info_both(' STARTING 12c SINGLE loop PURGE      **',write_file);
      log_info_both('***************************************',write_file);
      log_info_both('***************************************',write_file);
      log_info_both('min_creation_date => '
         ||TO_CHAR(min_creation_date,'DD/MON/YYYY:HH24/MI'),write_file);
      log_info_both('max_creation_date => '
         ||TO_CHAR(max_creation_date,'DD/MON/YYYY:HH24/MI'),write_file);
      if batch_size < 1 then v_batch_size := 20000; end if;
      log_info_both('batch_size => '||v_batch_size,write_file);
      -- if max_runtime < 1 then 
      --   v_max_runtime := 60; 
      --   stoptime := sysdate + NVL(v_max_runtime,24*60)/(24*60);
      -- end if;
      -- log_info('max_runtime => '||v_max_runtime);
      log_info_both('retention_period => '
         ||TO_CHAR(retention_period,'DD/MON/YYYY:HH24/MI'),write_file);
      log_info_both('composite_name => '||composite_name,write_file);
      log_info_both('composite_revision => '||composite_revision,write_file);
      log_info_both('soa_partition_name => '||soa_partition_name,write_file);

      log_info_both('procedure soa.delete_instances',write_file);
      log_info_both('time check',write_file);
      log_info_both('sysdate  => ' ||TO_CHAR(sysdate,'DD/MON/YYYY:HH24/MI'),write_file);
      log_info_both('stoptime => ' ||TO_CHAR( stoptime,'DD/MON/YYYY:HH24/MI'),write_file);
      if inmemory then
         log_info_both('inmemory => TRUE ',write_file);
      else
         log_info_both('inmemory => FALSE ',write_file);
      end if;
      if ignore_state then
         log_info_both('ignore_state is true, data will be purged irrespective of state ');
      end if;
     
      if compbased then
         log_info_both('compbased is true, flows will be purged based on component states ');
      end if; 

      IF (sysdate >= stoptime) THEN
            purge_loop := FALSE;
      END IF;

      if sql_trace then
        v_stmt := 'alter session set max_dump_file_size = unlimited';
        log_info(v_stmt);
        execute immediate v_stmt;
        v_stmt := 'alter session set events '
        || '''10046 trace name context forever, level 12''';
        log_info(v_stmt);
        execute immediate v_stmt;
      end if;

      select soa_purge_seq.nextval into soa_purge_job_no from dual;
      insert into soa_purge_history
        (job_no,start_time,end_time,type,thread,status)
      values
        (soa_purge_job_no,systimestamp,null,'S',0,'R');
      commit;

      execute immediate 'truncate table ' || prune_running_insts_table;
      execute immediate 'truncate table edn_log_messages';
      execute immediate 'delete from SOA_PURGE_LOG where LOG_DATE < sysdate - 30';

      componentPartInfo := getComponentPartitionInfo();
      composite_dn := make_composite_dn
                   (soa_partition_name, composite_name, composite_revision);

      -- Main loop.
      WHILE purge_loop = TRUE LOOP

      log_info_both('loop  count = ' || iterationCount,write_file);
      log_info_both('deleting instances',write_file);

      flow_id_sql := ' select s.flow_id from sca_flow_instance s where ' ;

      if ignore_state = false AND compbased = false then
         flow_id_sql := flow_id_sql
         || ' s.active_component_instances = 0 and ';
      end if;

      if inmemory then
         flow_id_sql :=  flow_id_sql
         || ' s.purgeable = ''Y'' and '
         || ' s.updated_time <= '
         || '''' || retention_period || '''' || ' and ';
      else
         if min_creation_date is not null then
            flow_id_sql :=  flow_id_sql 
            || ' s.created_time >= ' 
            || '''' || min_creation_date || '''' || ' and '; 
         end if;
         if max_creation_date is not null then 
            flow_id_sql :=  flow_id_sql 
            || ' s.created_time <= ' 
            || '''' || max_creation_date || '''' || ' and ';
         end if;
         if retention_period is not null then
            v_retention_period := retention_period;
            if max_creation_date is not null then
            -- placed for backward compatibility.
               if retention_period < max_creation_date then
                  v_retention_period := max_creation_date;
               end if;
            end if;
            flow_id_sql :=   flow_id_sql 
            || ' s.updated_time <= ' 
            || '''' || v_retention_period || '''' || ' and ';
         end if;
      end if;

      if composite_dn is not null then
         flow_id_sql := flow_id_sql
         || ' s.flow_id in (select c.flow_id from sca_flow_to_cpst c, sca_entity e, sca_partition p where c.composite_sca_entity_id = e.id '
         || composite_dn || ') and ';
      end if;

      flow_id_sql := flow_id_sql 
         || ' s.flow_id not in (select r.flow_id from temp_prune_running_insts r) and ';
      -- rownum is mandatory.
      flow_id_sql := flow_id_sql || ' rownum <= ' || v_batch_size ;
      flow_id_sql := 'insert into '||purge_id_table || flow_id_sql;
      execute immediate 'truncate table ' || purge_id_table;
      execute immediate flow_id_sql;
      total_rows := SQL%ROWCOUNT;
      debug_purge_both(purge_id_table,'Inserted = ',write_file);
      commit;
      IF total_rows = 0  THEN
           purge_loop := FALSE ;
      END IF ;
      
      -- if not compbased job and not ignoring state
      IF purge_loop AND ignore_state = false AND compbased = false then
        execute immediate 'insert into temp_prune_running_insts '
         || ' select flow_id from temp_flowid_purge p '
         || 'where exists '
         ||  '(select ''1'' '
         ||  'from sca_flow_assoc a '
         ||  'where a.correlated_flow_id = p.flow_id '
         ||  'and exists '
         ||   '(select ''1'' '
         ||   'from sca_flow_assoc a1, sca_flow_instance f '
         ||   'where a1.owner_flow_id = a.owner_flow_id '
         ||   'and a1.correlated_flow_id = f.flow_id '
         ||   'and f.active_component_instances > 0))';

         delete from temp_flowid_purge
         where flow_id in ( select flow_id from temp_prune_running_insts);

         total_rows := SQL%ROWCOUNT;
         debug_purge_both(purge_id_table,'Correlated flows pruned  = ',write_file);
         commit;
      end if;
       
	-- do pruning based on component states if compbased job
       if compbased then
         SELECT count(*) INTO total_rows_after_pruning FROM temp_flowid_purge;
         log_info_both('compbased is true, pruneOpenFlowIDs  , total flows before pruning = ' || total_rows_after_pruning);
         
	 pruneOpenFlowIDs (purge_id_table,prune_running_insts_table);

         delete from temp_flowid_purge
               where flow_id in ( select flow_id from temp_prune_running_insts);
	 debug_purge_both(purge_id_table,'Flows pruned  = ',write_file);

       end if;

      SELECT count(*) INTO total_rows_after_pruning FROM temp_flowid_purge;
      log_info_both('total purgeable flows after pruning = ' || total_rows_after_pruning);

      commit;

      IF purge_partitioned_component = false AND  ( componentPartInfo.fabricPartitioned='Y' OR componentPartInfo.fabricPartitioned='P') then
          log_info_both('inserting processed flow_id into temp_prune_running_insts',write_file);
          insert into temp_prune_running_insts select flow_id from temp_flowid_purge;
          debug_purge_both('temp_prune_running_insts','inserted = ',write_file);
          commit;
      END IF;

      delete_flow_instances
          (purge_id_table,
           purge_partitioned_component,
           componentPartInfo, 
           stoptime, 
           v_batch_size,
           min_creation_date, 
           max_creation_date, 
           retention_period,
           write_file);         

      log_info_both('time check',write_file);
      log_info_both('sysdate  = ' ||TO_CHAR(sysdate,'DD/MON/YYYY:HH24/MI'),write_file);
      log_info_both('stoptime = ' ||TO_CHAR( stoptime,'DD/MON/YYYY:HH24/MI'),write_file);
      IF (sysdate >= stoptime) THEN
            purge_loop := FALSE;
      END IF;
      iterationCount := iterationCount + 1;
       
      END LOOP;

     execute immediate 'truncate table ' || prune_running_insts_table;

      /* Note: The delete_stat_alert_tables is used by Single/Parallel purge.
               For single loop the batch_size is passed twice as it
               represents max_count and batch_size.
               The Single Loop purge will only delete the Statistics tables
               if there is time but the Parallel purge will do this in
               parallel to the spawned threads.
       */
      delete_stat_alert_tables
           (stoptime,
            v_batch_size,
            v_batch_size,
            retention_period,
            write_file);

     update soa_purge_history set 
         end_time = systimestamp,
       status = 'C'
      where job_no = soa_purge_job_no
        and type = 'S';
     commit;

    log_info_both('delete_soa_instances completed successfully',write_file);
 
    close_file(write_file);
    -- Set ACTION_NAME on SESSION to NULL. 
    DBMS_APPLICATION_INFO.SET_MODULE(MODULE_NAME => NULL, ACTION_NAME => NULL);

 EXCEPTION
  when others then
   close_file(write_file);
   DBMS_APPLICATION_INFO.SET_MODULE(MODULE_NAME => NULL, ACTION_NAME => NULL);
   log_error_both('ERROR(delete_soa_instances) ',write_file);     
   raise;

 end delete_soa_instances;

PROCEDURE deleteOSBResequencerInstances(
                   p_batch_size NUMBER DEFAULT 20000,
                   p_min_creation_date IN TIMESTAMP,
                   p_max_creation_date IN TIMESTAMP) is

       begin
         soa_osb.deleteOSBResequencerInstances(p_batch_size,p_min_creation_date,p_max_creation_date);
       end deleteOSBResequencerInstances;



FUNCTION lookupCompositeQuery (
    min_creation_date             IN   TIMESTAMP,
    max_creation_date             IN   TIMESTAMP,
	flow_ids                      IN   VARCHAR2 DEFAULT NULL,
    retention_period              IN   TIMESTAMP DEFAULT NULL,      
    composite_name                IN   VARCHAR2 DEFAULT NULL,
    composite_revision            IN   VARCHAR2 DEFAULT NULL,
	ecID                          IN   VARCHAR2 DEFAULT NULL,
	comp_state                    IN   VARCHAR2 DEFAULT NULL,
	instance_id                   IN   VARCHAR2 DEFAULT NULL,
    soa_partition_name            IN   VARCHAR2 DEFAULT NULL
	--inmemory 					  IN   BOOLEAN  DEFAULT FALSE,
	--compbased 			      IN   BOOLEAN  DEFAULT FALSE,
	--ignore_state 				  IN   BOOLEAN  DEFAULT FALSE
	
   ) RETURN VARCHAR2 IS
    composite_dn       VARCHAR2(2000);
	notNullConditionFound  BOOLEAN := FALSE;
    finallookupquery   VARCHAR2(6000) := '   ';
BEGIN
    --check if state is supplied , If its OPEN then STATE_UNKNOWN = -1, STATE_RUNNING = 0 , STATE_RECOVERY_REQUIRED = 1, STATE_FAULTED = 3,STATE_SUSPENDED = 5,   @Deprecated int STATE_IN_FLIGHT ( this is not required as depricated)


	IF comp_state IS NOT NULL THEN	    
        IF comp_state = 'OPEN' THEN 
            finallookupquery := finallookupquery || ' cmpt.state in (-1,0,1,3,5)';
            notNullConditionFound := TRUE;
           ELSE  IF comp_state <> 'ALL' THEN 
            dbms_output.put_line('Status input ignored : Only ALL or OPEN status allowed');
            END IF;
        END IF;       
	END IF;

	IF flow_ids IS NOT NULL THEN
		IF notNullConditionFound THEN 
			finallookupquery := finallookupquery || ' and cmpt.flow_id IN (' || flow_ids || ')'; 				
		ELSE 		
			finallookupquery := finallookupquery || ' cmpt.flow_id IN (' || flow_ids || ')';  
		END IF;
		notNullConditionFound := TRUE;
	ELSE   
		IF instance_id IS NOT NULL  OR ecID IS NOT NULL  THEN    
			IF notNullConditionFound THEN 
				IF instance_id IS NOT NULL THEN
					finallookupquery := finallookupquery || ' and cmpt.ID IN (' || instance_id || ')';                    
				ELSE
					 finallookupquery := finallookupquery || ' and cmpt.ecID IN(select regexp_substr('''||ecID || ''', ''[^,]+'', 1, level) from dual  connect by regexp_substr(''' || ecID || ''', ''[^,]+'', 1, level)  is not null )';

				END IF;
			ELSE 
				IF instance_id IS NOT NULL THEN
					finallookupquery := finallookupquery || '  cmpt.ID IN (' || instance_id || ')';                    
				ELSE
					finallookupquery := finallookupquery || '  cmpt.ecID IN(select regexp_substr('''||ecID || ''', ''[^,]+'', 1, level) from dual  connect by regexp_substr(''' || ecID || ''', ''[^,]+'', 1, level)  is not null )';
				END IF;
			END IF;
			notNullConditionFound := TRUE;
		END IF;
	END IF;

	IF retention_period IS NOT NULL THEN
		IF notNullConditionFound THEN
			finallookupquery := finallookupquery || ' and to_timestamp(to_char(cmpt.CREATED_TIME)) <= ' || '''' || retention_period || '''';
		ELSE 
			finallookupquery := finallookupquery || '  to_timestamp(to_char(cmpt.CREATED_TIME)) <= ' || '''' || retention_period || '''';
		END IF;
		notNullConditionFound := TRUE;
	END IF;
	IF min_creation_date IS NOT NULL THEN
	  IF notNullConditionFound THEN
			finallookupquery := finallookupquery || ' and cmpt.CREATED_TIME >= ' || '''' || min_creation_date || '''';
	   ELSE 
			finallookupquery := finallookupquery || '  cmpt.CREATED_TIME >= ' || '''' || min_creation_date || '''';
	   END IF;
	   notNullConditionFound := TRUE;
	END IF;
	IF max_creation_date IS NOT NULL THEN
		 IF notNullConditionFound THEN
			finallookupquery := finallookupquery || ' and cmpt.CREATED_TIME <= ' || '''' || max_creation_date || '''';
		ELSE 
			finallookupquery := finallookupquery || '  cmpt.CREATED_TIME <= ' || '''' || max_creation_date || '''';
		END IF;
		notNullConditionFound := TRUE;
	END IF;

	/*IF inmemory THEN
         finallookupquery := finallookupquery
         || ' and cmpt.purgeable = ''Y'' and '
         || ' cmpt.updated_time <= '
         || '''' || retention_period || '''';
	 END IF; 
	 */

	 /*IF ignore_state = FALSE AND compbased = FALSE then
         finallookupquery := finallookupquery
         || ' and cmpt.active_component_instances = 0 ';
      end if;
	*/

    --finallookupquery := finallookupquery || ' ORDER BY  cmpt.created_time ASC ';
    dbms_output.put_line('Final query to fetch all composite instances' || finallookupquery);
    IF notNullConditionFound THEN
         RETURN finallookupquery;
    ELSE 
         RETURN NULL;
         END IF;
EXCEPTION
    WHEN OTHERS THEN
        dbms_output.put_line('Error while all composite lookup');
        RETURN finallookupquery;
END lookupCompositeQuery;

FUNCTION abort_components( min_creation_date IN TIMESTAMP, max_creation_date IN TIMESTAMP, lookupcompquery IN VARCHAR2 , batch_size IN INTEGER, cmpst_name IN VARCHAR2, composite_dn IN VARCHAR2, max_runtime IN INTEGER, update_retention IN BOOLEAN) RETURN INTEGER AS
    v_totalcount    INTEGER := 0;
	batch_total_count   INTEGER := 0;
	common_abort_query VARCHAR2(4000);
    lookupWithCompName VARCHAR2(4000);
BEGIN
	batch_total_count := soa_orabpel.abortComponentInstances(min_creation_date, max_creation_date, lookupcompquery, batch_size,cmpst_name,max_runtime, update_retention);
	 IF composite_dn IS NOT NULL THEN
        	lookupWithCompName := ' cmpt.flow_id in (select c.flow_id from sca_flow_to_cpst c, sca_entity e, sca_partition p where c.composite_sca_entity_id = e.id '
         || composite_dn || ')';
	 END IF;
     IF lookupcompquery IS NOT NULL THEN
            IF lookupWithCompName IS NOT NULL THEN
               lookupWithCompName := lookupcompquery || 'AND' || lookupWithCompName;
            ELSE
               lookupWithCompName := lookupcompquery;
    		END IF;
      END IF;
	 
	 dbms_output.put_line('Final query : '|| lookupWithCompName);

	batch_total_count := soa_mediator.abortComponentInstances(lookupWithCompName, batch_size,cmpst_name,max_runtime);
	batch_total_count := soa_workflow.abortComponentInstances(lookupWithCompName, batch_size,cmpst_name,max_runtime);
	--AbortCommonFaultActivity
	common_abort_query := 'update sca_common_fault  scf set scf.state = 1 WHERE scf.state not in (11,1,10,14) AND scf.flow_id in (select flow_id from sca_flow_instance cmpt WHERE ' || lookupWithCompName || ')';
	dbms_output.put_line('sca_common_fault aborted...query is :: '|| common_abort_query);

	EXECUTE IMMEDIATE  common_abort_query;
	COMMIT;
	dbms_output.put_line('sca_common_fault aborted...');

	--AbortAsyncCpntStateActivity
	common_abort_query := 'update sca_async_cpnt_state acps  set acps.state = 4 WHERE acps.state <> 2  AND acps.flow_id in (select flow_id from sca_flow_instance cmpt WHERE ' || lookupWithCompName || ')';
	dbms_output.put_line('sca_async_cpnt_state aborted..with query ::'|| common_abort_query);
	EXECUTE IMMEDIATE  common_abort_query;
	COMMIT;
	--AbortFlowToCpstActivity
	common_abort_query := 'update  sca_flow_to_cpst ftcp  set ftcp.admin_state= 2, ftcp.active_component_instances = 0  WHERE ftcp.flow_id in (select flow_id from sca_flow_instance cmpt WHERE ' || lookupWithCompName || ')';
	dbms_output.put_line('sca_flow_to_cpst aborted..with query ::'|| common_abort_query);
	EXECUTE IMMEDIATE  common_abort_query;
	COMMIT;

	--AbortRejectedMessageActivity
	common_abort_query := 'update  sca_rejected_message srm  set srm.srm_state = 13  WHERE srm.srm_state not in(11,1,10,14) AND srm.flow_id in (select flow_id from sca_flow_instance cmpt WHERE ' || lookupWithCompName || ')';
	EXECUTE IMMEDIATE  common_abort_query;
	COMMIT;
	dbms_output.put_line('sca_rejected_message aborted..query is ::'|| common_abort_query);

	common_abort_query := 'update  sca_flow_instance cmpt set cmpt.UNHANDLED_FAULTS = 0, cmpt.RECOVERABLE_FAULTS = 0, cmpt.ACTIVE_COMPONENT_INSTANCES = 0, cmpt.ADMIN_STATE= 2, cmpt.UPDATED_TIME = SYSDATE WHERE ' || lookupWithCompName;
	EXECUTE IMMEDIATE  common_abort_query;
	COMMIT;
	dbms_output.put_line('sca_flow_instance  aborted as per query...' || common_abort_query);


    RETURN v_totalcount;
    EXCEPTION
    WHEN OTHERS THEN
       dbms_output.put_line('Error while abort scripts for flow_instances and related tables');
        RETURN -1;
END abort_components;




PROCEDURE abortcompositeinstances (
        min_creation_date             IN   TIMESTAMP,
        max_creation_date             IN   TIMESTAMP,
		flow_ids                       IN   VARCHAR2 DEFAULT NULL,
        batch_size                    IN   INTEGER DEFAULT 20000,
        max_runtime                   IN   INTEGER DEFAULT 1200,
		--stoptime                      IN   DATE DEFAULT NULL, I see this change in 12C delete script , need to check with Binu/Shereef
	--instance_id                   IN   VARCHAR2, 
        retention_period              IN   TIMESTAMP DEFAULT NULL,
        composite_name                IN   VARCHAR2 DEFAULT NULL,
        composite_revision            IN   VARCHAR2 DEFAULT NULL,
		ec_id                         IN   VARCHAR2 DEFAULT NULL,
        soa_partition_name            IN   VARCHAR2 DEFAULT NULL,
        composite_state               IN   VARCHAR2 DEFAULT NULL,
        update_retention              IN   BOOLEAN DEFAULT true,
		--newly added fields in 12C
		--PQS 						  IN   INTEGER DEFAULT 1,  Present in 12C deleteSoaInstances but its not been used. 
        sql_trace 					  IN   BOOLEAN DEFAULT false--fine
      --inmemory 					  IN   BOOLEAN DEFAULT false, --not needed                 
	  --compbased 					  IN   BOOLEAN DEFAULT false,--not require
	  --ignore_state 				  IN   BOOLEAN DEFAULT false--not needed
		
    ) IS

        lookupcompquery      VARCHAR2(4000);
		total_rows           INTEGER := 0;
        dir_path             VARCHAR2(4000);
        f_name           	 VARCHAR2(400);
        lastSlashIndex       NUMBER;
        l_file               utl_file.file_type;
        l_text               VARCHAR2(32767);
        l_min_creation_date  TIMESTAMP;
        l_max_creation_date  TIMESTAMP;
        l_batch_size         INTEGER ;
        l_max_runtime        INTEGER ;
		l_instance_id        INTEGER;
		instance_id          VARCHAR2(4000);
        l_retention_period   TIMESTAMP;
        l_composite_name     VARCHAR2(1024) ;
        l_composite_revision VARCHAR2(1024) ;
		l_ec_id              VARCHAR2(1024) ;
        l_soa_partition_name VARCHAR2(1024) ;
        l_composite_state    VARCHAR2(1024) ;
        l_keep_workflow_inst BOOLEAN ;
        file_path            VARCHAR2(400);
		v_stmt               VARCHAR2(500);
		composite_dn		 VARCHAR2(500);

    BEGIN
    instance_id := NULL;
    file_path := NULL;
	  /* IF file_path IS NOT NULL THEN
			lastSlashIndex := INSTR(file_path, '/', -1);
			dir_path :=  substr(file_path , 0, lastSlashIndex);
			f_name := substr(file_path, lastSlashIndex+1, 1024);
             dbms_output.put_line(dir_path);
            dbms_output.put_line(f_name);
			createdirectory('USER_DIR' , dir_path);
			l_file := utl_file.fopen('USER_DIR', f_name, 'R', 32767);
			LOOP
				utl_file.get_line(l_file, l_text, 32767);
                   dbms_output.put_line(l_text);
				--To check if this is header
				IF instr( regexp_substr(l_text, '[^,]+', 1, 1), 'MIN_CREATION_DATE')  > 0 THEN
					CONTINUE;
				END IF;
                l_min_creation_date   :=  CASE WHEN  regexp_substr(l_text, '[^,]+', 1, 1)  = 'NULL' THEN NULL  ELSE  to_timestamp(regexp_substr(l_text, '[^,]+', 1, 1),'YYYY-MM-DD') END ;
				l_max_creation_date   := CASE WHEN  regexp_substr(l_text, '[^,]+', 1, 2)  = 'NULL'  THEN NULL  ELSE  to_timestamp(regexp_substr(l_text, '[^,]+', 1, 2),'YYYY-MM-DD') END ;
				l_batch_size          := CASE WHEN  regexp_substr(l_text, '[^,]+', 1,3)   = 'NULL'   THEN NULL ELSE TO_NUMBER(regexp_substr(l_text, '[^,]+', 1,3)) END;
				l_max_runtime         := CASE WHEN  regexp_substr(l_text, '[^,]+', 1, 4) = 'NULL'   THEN NULL ELSE TO_NUMBER(regexp_substr(l_text, '[^,]+', 1, 4)) END;
				l_instance_id         := CASE WHEN  regexp_substr(l_text, '[^,]+', 1, 5) = 'NULL'   THEN NULL ELSE TO_NUMBER(regexp_substr(l_text, '[^,]+', 1, 5)) END;
				l_retention_period    := CASE WHEN  regexp_substr(l_text, '[^,]+', 1, 6)  = 'NULL' THEN NULL ELSE  to_timestamp(regexp_substr(l_text, '[^,]+', 1, 6),'YYYY-MM-DD') END ;
				l_composite_name           := CASE WHEN regexp_substr(l_text, '[^,]+', 1, 8) = 'NULL' THEN NULL ELSE regexp_substr(l_text, '[^,]+', 1, 8) END;
				l_composite_revision       := CASE WHEN regexp_substr(l_text, '[^,]+', 1, 9) = 'NULL' THEN NULL ELSE regexp_substr(l_text, '[^,]+', 1, 9) END;
				l_ec_id                    := CASE WHEN regexp_substr(l_text, '[^,]+', 1, 10) = 'NULL' THEN NULL ELSE regexp_substr(l_text, '[^,]+', 1, 10) END;
				l_soa_partition_name       := CASE WHEN regexp_substr(l_text, '[^,]+', 1,11) = 'NULL' THEN NULL ELSE regexp_substr(l_text, '[^,]+', 1,11) END;
				l_composite_state          := CASE WHEN regexp_substr(l_text, '[^,]+', 1, 12) = 'NULL' THEN NULL ELSE regexp_substr(l_text, '[^,]+', 1, 12) END;
				IF  regexp_substr(l_text, '[^,]+', 1, 13) = 'true' THEN
					l_keep_workflow_inst     := true;
				ELSE
					l_keep_workflow_inst     := false;
				END IF ;

				IF l_composite_state IS NOT NULL AND (composite_state != 'ALL' OR  composite_state != 'OPEN' ) THEN
					dbms_output.put_line('Invalid arg composite state, expected OPEN or ALL');
					return ;
				END IF;
				IF keep_workflow_inst THEN
					--Need to check with as this condition is specific to purge but not for abort case
					dbms_output.put_line('keep_workflow_inst is true, workflow data will not be purged by this run ');
				END IF;


			lookupcompquery := lookupcompositequery( l_min_creation_date, l_max_creation_date, l_retention_period, l_composite_name, l_composite_revision, l_ec_id, l_composite_state, l_instance_id,l_soa_partition_name);
			total_rows := abort_components(lookupcompquery, batch_size);

			END LOOP;
			utl_file.fclose(l_file);
        ELSE
        */
		
		IF sql_trace THEN
			v_stmt := 'alter session set max_dump_file_size = unlimited';
			log_info(v_stmt);
			EXECUTE IMMEDIATE v_stmt;
			v_stmt := 'alter session set events '
			|| '''10046 trace name context forever, level 12''';
			log_info(v_stmt);
			EXECUTE IMMEDIATE v_stmt;
       END IF;
		
			lookupcompquery := lookupcompositequery( min_creation_date, max_creation_date,flow_ids , retention_period, composite_name, composite_revision, ec_id,composite_state, instance_id, soa_partition_name);
            composite_dn := make_composite_dn(soa_partition_name, composite_name, composite_revision);
			IF lookupcompquery IS NULL AND composite_name IS NULL
            THEN 
                dbms_output.put_line('None of search criterias are matched , abord skipped');
                return;
            END IF;
		total_rows := abort_components(min_creation_date, max_creation_date, lookupcompquery, batch_size, composite_name, composite_dn, max_runtime, update_retention);
       --END IF;
    EXCEPTION
       WHEN no_data_found THEN
        NULL;
        WHEN OTHERS THEN
            dbms_output.put_line('Abort operation failed try again, contact support if problem persists');
            RAISE;
END abortcompositeinstances;

