/*
Rem
Rem
Rem Copyright (c) 2012, 2020, Oracle and/or its affiliates. 
Rem All rights reserved.
Rem
Rem    NAME
Rem      delete_instances_wrapper.sql 
Rem
Rem    DESCRIPTION
Rem
Rem    NOTES
Rem      <other useful comments, qualifications, etc.>
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    apfwkr      11/04/20 - Backport
Rem                           apfwkr_blr_backport_31572611_12.2.1.4.200917soabp
Rem                           from st_pcbpel_12.2.1.4.0soabp
Rem    apfwkr      10/19/20 - Backport apfwkr_blr_backport_31572611_12.2.1.4.0
Rem                           from main
Rem    apfwkr      10/16/20 - Backport apfwkr_blr_backport_31572611_12.2.1.3.0
Rem                           from main
Rem    apfwkr      10/15/20 - Backport
Rem                           shabdull_blr_backport_31572611_12.2.1.2.0 from
Rem                           st_pcbpel_12.2.1.2.0
Rem    shabdull    07/16/20 - Backport linlxu_bug-31572611_12212 from
Rem                           st_pcbpel_12.2.1.2.0
Rem    mbousamr    09/12/12 - Created
Rem
*/

/*
*  is_11g_required
*  Determine if the 11g purge is required. 
*/
 FUNCTION is_11g_required (
                   min_creation_date in timestamp,
                   max_creation_date in timestamp,
                   upgrade_date in timestamp)
   return boolean
   is

  begin 

     if upgrade_date is null
      then
        return FALSE;
     else
        if min_creation_date is not NULL then
           if min_creation_date > upgrade_date then
              return FALSE;
           end if;
        end if;
     end if;

     return TRUE;

 end is_11g_required;           
              

/*
*  is_purge_running
*  Determine if an 11g or 12c purge is running.
*/
 FUNCTION is_purge_running 
   return boolean
   is

    CURSOR c_gvsession is
    select inst_id, sid, serial#
     from gv$session
    where action = 'SOAPURGEEXECUTING';

  begin

      FOR gvsess_rec in c_gvsession
      LOOP
          RETURN TRUE;
      END LOOP;

      RETURN FALSE;

  end is_purge_running;


/*
*  delete_instances
*  Wrapper for SINGLE LOOP purge.
*  Executes both 12c and 11g purge if required.
*/

 procedure delete_instances ( min_creation_date in timestamp,
                   max_creation_date in timestamp,
                   batch_size in integer default 20000,
                   max_runtime in integer default 60,
                   retention_period in timestamp default null,
                   purge_partitioned_component in boolean default false,
                   composite_name in  varchar2 default null,
                   composite_revision in varchar2 default null,
                   soa_partition_name in varchar2 default null,
                   ignore_state in boolean default false,
                   PQS in integer default 1,
                   sql_trace IN boolean default false,
                   inmemory IN boolean default false,
		   compbased IN boolean default false
                   ) is


    run_11g boolean := TRUE;
    upgrade_date_v timestamp := NULL;
    max_runtime_v number := max_runtime;
    stoptime_d date := NULL;
    wait_for_11g boolean := TRUE;

 begin

    SELECT soa_upgrade_date
      INTO upgrade_date_v
      FROM version
     WHERE ROWNUM < 2;

    run_11g := is_11g_required(
                     min_creation_date,
                     max_creation_date,
                     upgrade_date_v);
    log_info_both('************ upgrade_date_v '||upgrade_date_v);

    if max_runtime < 1 then
       max_runtime_v := 60;
    end if;
    stoptime_d := sysdate + NVL(max_runtime_v,24*60)/(24*60);

    IF run_11g
     THEN
       log_info_both('************ run_11g true ' );

       IF min_creation_date is null
       OR max_creation_date is null then
          log_info('********************************************************');
          log_info('*** WARNING:                                         ***');
          log_info('*** ========                                         ***');
          log_info('*** The 11g Purge requires the MINIMUM CREATION DATE ***');
          log_info('***   and MAXIMUM CREATION DATE                      ***');
          log_info('*** The 11g Purge will NOT BE EXECUTED.              ***');
          log_info('***   Please specify the MIN and MAX creation dates  ***');
          log_info('***   until all 11g data has been removed.           ***');
          log_info('********************************************************');
       ELSE
         -- Call 11g
         soa11g.delete_instances (
             min_creation_date => min_creation_date,
             max_creation_date => max_creation_date,
             batch_size => batch_size,
             stoptime => stoptime_d,
             retention_period => retention_period,
             purge_partitioned_component => purge_partitioned_component,
             composite_name => composite_name,
             composite_revision => composite_revision,
             soa_partition_name  => soa_partition_name,
             ignore_state => ignore_state,
             sql_trace => sql_trace);

       END IF;

       IF (sysdate < stoptime_d) THEN
           -- call 12c if time permits.
           soa.delete_soa_instances (
               min_creation_date => min_creation_date,
               max_creation_date => max_creation_date,
               batch_size => batch_size,
               stoptime => stoptime_d,
               retention_period => retention_period,
               purge_partitioned_component => purge_partitioned_component,
               composite_name => composite_name,
               composite_revision => composite_revision,
               soa_partition_name => soa_partition_name,
               ignore_state => ignore_state,
               PQS => PQS,
               sql_trace => sql_trace,
               inmemory => inmemory,
	       compbased => compbased);

       END IF;

    ELSE
       -- call 12c 
       soa.delete_soa_instances (
             min_creation_date => min_creation_date,
             max_creation_date => max_creation_date,
             batch_size => batch_size,
             stoptime => stoptime_d,
             retention_period => retention_period,
             purge_partitioned_component => purge_partitioned_component,
             composite_name => composite_name,
             composite_revision => composite_revision,
             soa_partition_name => soa_partition_name,
             ignore_state => ignore_state,
             PQS => PQS,
             sql_trace => sql_trace,
             inmemory => inmemory,
             compbased => compbased);

    END IF;

    EXCEPTION
     WHEN NO_DATA_FOUND
      THEN
      log_info
       ('*** ERROR - UPGRADE DATE NOT FOUND IN SOA SCHEMA TABLE VERSION ***');
      log_info
       ('***         SOA PURGE ABORTED CONTACT ORACLE SUPPORT  ***');
      RAISE_APPLICATION_ERROR
       (-20200, 'ERROR - Upgrade date not found in SOA Schema table VERSION.');

  end delete_instances;
       
/*
*  delete_instances_in_parallel
*  Wrapper for PARALLEL LOOP purge.
*  Executes both 12c and 11g purge if required.
*/
  procedure delete_instances_in_parallel (
                   min_creation_date in timestamp,
                   max_creation_date in timestamp,
                   batch_size in integer default 20000,
                   max_runtime in integer default 60,
                   retention_period in timestamp default null,
                   DOP in integer default 4,
                   max_count in integer default 1000000,
                   purge_partitioned_component in boolean default false,
                   composite_name in  varchar2 default null,
                   composite_revision in varchar2 default null,
                   soa_partition_name in varchar2 default null,
                   ignore_state in boolean default false,
                   PQS in integer default 1,
                   sql_trace IN boolean default false,
                   inmemory IN boolean default false,
		   compbased IN boolean default false
                ) is

    run_11g boolean := TRUE;
    upgrade_date_v timestamp := NULL;
    max_runtime_v number := max_runtime;
    stoptime_v varchar2(100) := NULL;
    stoptime_d date := NULL;
    wait_for_11g boolean := TRUE;    

 begin

    SELECT soa_upgrade_date
      INTO upgrade_date_v
      FROM version
     WHERE ROWNUM < 2;

    run_11g := is_11g_required(
                     min_creation_date,
                     max_creation_date,
                     upgrade_date_v);

    if max_runtime < 1 then
       max_runtime_v := 60;
    end if;
    stoptime_v :=
     to_char(sysdate + NVL(max_runtime_v,24*60)/(24*60),'DD/MON/YYYY:HH24/MI');
    stoptime_d := to_date(stoptime_v,'DD/MON/YYYY:HH24/MI');

    IF run_11g
     THEN

       IF min_creation_date is null
       OR max_creation_date is null then
          log_info('********************************************************');
          log_info('*** WARNING:                                         ***');
          log_info('*** ========                                         ***');
          log_info('*** The 11g Purge requires the MINIMUM CREATION DATE ***');
          log_info('***   and MAXIMUM CREATION DATE                      ***');
          log_info('*** The 11g Purge will NOT BE EXECUTED.              ***');
          log_info('***   Please specify the MIN and MAX creation dates  ***');
          log_info('***   until all 11g data has been removed.           ***');
          log_info('********************************************************');
       ELSE
         -- Call 11g
         soa11g.delete_instances_in_parallel (
             min_creation_date => min_creation_date,
             max_creation_date => max_creation_date,
             batch_size => batch_size,
             stoptime => stoptime_v,
             retention_period => retention_period,
             DOP => DOP,
             max_count => max_count,
             purge_partitioned_component => purge_partitioned_component,
             composite_name => composite_name,
             composite_revision => composite_revision,
             soa_partition_name => soa_partition_name,
             ignore_state => ignore_state,
             sql_trace => sql_trace);

         wait_for_11g := TRUE;
         WHILE wait_for_11g
           AND (sysdate < stoptime_d)
           LOOP
             log_info('*** LOOP AND WAIT 60secs FOR 11g PARALLEL PURGE ***');
             DBMS_LOCK.sleep(seconds => 60);
             wait_for_11g := is_purge_running;
         END LOOP;

       END IF;

       IF (sysdate < stoptime_d) THEN
          -- call 12c if time permits.
          soa.delete_soa_inst_in_parallel (
              min_creation_date => min_creation_date,
              max_creation_date => max_creation_date,
              batch_size => batch_size,
              stoptime => stoptime_v,
              retention_period => retention_period,
              DOP => DOP,
              max_count => max_count,
              purge_partitioned_component => purge_partitioned_component,
              composite_name => composite_name,
              composite_revision => composite_revision,
              soa_partition_name => soa_partition_name,
              ignore_state => ignore_state,
              PQS => PQS,
              sql_trace => sql_trace,
              inmemory => inmemory,
              compbased => compbased);
       END IF;

    ELSE
       -- call 12c
       soa.delete_soa_inst_in_parallel (
             min_creation_date => min_creation_date,
             max_creation_date => max_creation_date,
             batch_size => batch_size,
             stoptime => stoptime_v,
             retention_period => retention_period,
             DOP => DOP,
             max_count => max_count,
             purge_partitioned_component => purge_partitioned_component,
             composite_name => composite_name,
             composite_revision => composite_revision,
             soa_partition_name => soa_partition_name,
             ignore_state => ignore_state,
             PQS => PQS,
             sql_trace => sql_trace,
             inmemory => inmemory,
	     compbased => compbased);

    END IF;

    EXCEPTION
     WHEN NO_DATA_FOUND
      THEN
      log_info
       ('*** ERROR - UPGRADE DATE NOT FOUND IN SOA SCHEMA TABLE VERSION ***');
      log_info
       ('***         SOA PURGE ABORTED CONTACT ORACLE SUPPORT  ***');
      RAISE_APPLICATION_ERROR
       (-20200, 'ERROR - Upgrade date not found in SOA Schema table VERSION.');

  end delete_instances_in_parallel;
