create or replace 
PACKAGE adapter_migrate AS


    FUNCTION upgrade_rejected_messages(batch_size in number default 10000, stoptime  in date default null, max_count number default 0, 
                                        thread in integer default -1, dop in integer default 1) RETURN boolean;
    FUNCTION upgrade_rejected_message(rejected_message IN REJECTED_MESSAGE%ROWTYPE,thread INTEGER,  log_file IN utl_file.file_type) RETURN integer;

    FUNCTION get_rejected_messages_count(query in varchar2) RETURN NUMBER;

    PROCEDURE insert_rejected_message
    (
        ecid IN VARCHAR2, 
        srm_id IN NUMBER, 
        flow_id  IN NUMBER, 
        batch_id IN VARCHAR2,
        created_time IN timestamp,
        service_name IN VARCHAR2,
        binding_type IN VARCHAR2,
        protocol_message_id IN VARCHAR2, 
        protocol_correlation_id IN VARCHAR2,
        error_category IN VARCHAR2, 
        error_code IN VARCHAR2,
        error_message IN CLOB, 
        stack_trace IN CLOB,
        retry_count IN NUMBER,
        message_origin_reference IN VARCHAR2, 
        additional_properties IN VARCHAR2,
        tenant_id IN NUMBER,
        partition_id IN NUMBER
    );

    PROCEDURE update_rejected_message
    (
        in_id      IN NUMBER,
        in_flow_id IN NUMBER
    );

    PROCEDURE mark_rejected_message_error
    (
        in_id      IN NUMBER,
        ecid       IN VARCHAR,
        in_flowid  IN NUMBER,
        log_file IN utl_file.file_type         
    );



END adapter_migrate;
/

show errors

   
create or replace PACKAGE BODY adapter_migrate AS

    
    
    FUNCTION upgrade_rejected_messages(batch_size in number default 10000, stoptime  in date default null, max_count number default 0, 
                                        thread in integer default -1, dop in integer default 1) RETURN boolean 
    IS
    --CURSOR REJ_CUR_PARALLEL IS  SELECT * from REJECTED_MESSAGE WHERE FLOW_ID IS NULL AND MOD (DBMS_UTILITY.GET_HASH_VALUE(ECID,0,DOP ), DOP )= THREAD  ORDER BY CREATED_TIME DESC;
    TYPE REJ_TYPE IS TABLE OF REJECTED_MESSAGE%ROWTYPE;
    REJ_ARRAY REJ_TYPE;
    log_file utl_file.file_type;
    limit_val number :=  1000;
    ret_value integer := 0;
    rej_cursor sys_refcursor;
    rejected_message_query varchar2(500);
    rejected_message_count_query varchar2(500);
    main_query varchar2(60) := 'SELECT * from REJECTED_MESSAGE where FLOW_ID IS NULL ';
    main_count_query varchar2(75) := 'SELECT COUNT(1) from REJECTED_MESSAGE where FLOW_ID IS NULL AND ROWNUM =1 ';
    order_by_time_query varchar2(30) := ' ORDER BY CREATED_TIME DESC ';
    rownum_query varchar2(20) := ' and rownum <= ';
    parallel_query varchar2(200) :=' and mod (dbms_utility.get_hash_value(ecid,0,' || dop || '),' || dop || ')=' ||thread ;
    upgrade_done boolean := true;
    upgrade_done_str varchar2(10) := 'true'; 
    commit_size number :=  100;
    commit_size_count number;
    num_rej_part_count number;
    adapter_message_count number;
    BEGIN
        BEGIN
            dbms_output.enable(NULL);
            log_file := get_logfile();
            dumpScaEntities(log_file); --dump it at the begining
            log_debug('adapter_migrate.upgrade_rejected_messages start', log_file);
            select count(1) into adapter_message_count from REJECTED_MESSAGE WHERE FLOW_ID IS NULL;
            log_error('inside adapter_migrate.upgrade_rejected_messages total number of records to be processed = ' || adapter_message_count, log_file);
            --insert_error_log(thread, 'adapter_migrate.upgrade_rejected_messages',0, -20001, 'Total number of records remaining to be processed = ' || adapter_message_count);
            IF max_count = 0 THEN
                rejected_message_query := main_query;
                rejected_message_count_query := main_count_query;
            ELSE
                rejected_message_query := main_query ||rownum_query || max_count ;
                rejected_message_count_query := main_count_query ;
            END IF;
            if thread >= 0 then
                rejected_message_query := rejected_message_query || parallel_query ;
                rejected_message_count_query := rejected_message_count_query || parallel_query ;
            end if;
            rejected_message_query := rejected_message_query || order_by_time_query;
            log_debug('upgrade_rejected_messages query = ' ||rejected_message_query,log_file);
            --insert_error_log(thread, 'adapter_migrate.upgrade_rejected_messages',0, -20002, 'Rejected Message Query = ' || rejected_message_query);
           -- num_rej_part_count := get_rejected_messages_count(rejected_message_count_query);
            --insert_error_log(thread, 'adapter_migrate.upgrade_rejected_messages',0, -20003, 'Rejected Message COUNT Query = ' || rejected_message_count_query || 'Number of records to process by this instance =>' || num_rej_part_count);
            BEGIN
                log_debug('Opening cursor for rejected messages ', log_file);
                OPEN rej_cursor FOR rejected_message_query;
                LOOP
                    FETCH rej_cursor BULK COLLECT into REJ_ARRAY LIMIT batch_size;
                    IF REJ_ARRAY.COUNT = 0 THEN
                        log_debug('No records found in rejected_message', log_file);
                        CLOSE rej_cursor;
                        close_logfile(log_file);
                        COMMIT;
                        --insert_error_log(thread, 'adapter_migrate.upgrade_rejected_messages',0, -20004, 'No more records in the cursor, exiting');
                        num_rej_part_count := get_rejected_messages_count(rejected_message_count_query);
                        IF (num_rej_part_count > 0) THEN 
                            RETURN false;
                        ELSE 
                            RETURN true;
                        END IF;
                    END IF;
                    IF (sysdate >= stoptime) THEN
                        log_debug('Stop time reached. Rejected message records still left for upgrade',log_file);
                        CLOSE rej_cursor;
                        close_logfile(log_file);
                        COMMIT;
                        --insert_error_log(thread, 'adapter_migrate.upgrade_rejected_messages',0, -20005, 'Stop time reached, exiting...');
                        RETURN false;
                    END IF;
                    IF is_stop_injected THEN
                        log_debug('Stop injected. Rejected message records still left for upgrade',log_file);
                        CLOSE rej_cursor;
                        close_logfile(log_file);
                        COMMIT;
                        --insert_error_log(thread, 'adapter_migrate.upgrade_rejected_messages',0, -20006, 'Stop injected, exiting...');
                        RETURN false;
                    END IF;
                    commit_size_count := 1;
                    FOR I in 1..REJ_ARRAY.COUNT
                    LOOP
                        BEGIN
                            IF (sysdate >= stoptime) THEN
                                log_debug('Stop time reached. Rejected message records still left for upgrade',log_file);
                                CLOSE rej_cursor;
                                close_logfile(log_file);
                                COMMIT;
                                --insert_error_log(thread, 'adapter_migrate.upgrade_rejected_messages',0, -20007, 'Stop time reached while looping through cursor, exiting...');
                                return FALSE;
                            END IF;
                            IF is_stop_injected THEN
                                log_debug('Stop injected. Rejected message records still left for upgrade',log_file);
                                CLOSE rej_cursor;
                                close_logfile(log_file);
                                COMMIT;
                                --insert_error_log(thread, 'adapter_migrate.upgrade_rejected_messages',0, -20008, 'Stop injected while looping through cursor, exiting...');
                                return FALSE;
                            END IF;
                            ret_value := upgrade_rejected_message(REJ_ARRAY(I), thread,  log_file);
                            IF ret_value = 1 THEN -- WARNING
                                mark_rejected_message_error(REJ_ARRAY(I).ID,REJ_ARRAY(I).ECID,-9, log_file);
                            END IF;
                            IF ret_value = 2 THEN -- ERROR
                                mark_rejected_message_error(REJ_ARRAY(I).ID,REJ_ARRAY(I).ECID,-1, log_file);
                            END IF;
                            IF commit_size_count >= commit_size  THEN
                               COMMIT; 
                               commit_size_count := 1;
                            ELSE
                               commit_size_count := commit_size_count + 1;
                            END IF;
                        EXCEPTION
                            WHEN OTHERS THEN
                                insert_error_log(thread, 'adapter_migrate.upgrade_rejected_messages',2, -20028, 'Unable to process single rejected message for ' ||
                                                           REJ_ARRAY(I).ID || '  # ' || SQLCODE || '-' || SQLERRM);
                                mark_rejected_message_error(REJ_ARRAY(I).ID,REJ_ARRAY(I).ECID,-1, log_file);
                                COMMIT;
                        END;
                    END LOOP;
                    log_debug('Comitting after ' || batch_size || ' records', log_file);
                    COMMIT;
                END LOOP;
                log_debug('Closing cursor for rejected messages', log_file);
                CLOSE rej_cursor;
                COMMIT; -- Final commit
                --insert_error_log(thread, 'adapter_migrate.upgrade_rejected_messages',0, -20029, 'Final commit done');
                num_rej_part_count := get_rejected_messages_count(rejected_message_count_query);
                IF (num_rej_part_count > 0) THEN 
                    upgrade_done := false;
                    upgrade_done_str := 'false';
                ELSE 
                    upgrade_done := true;
                    upgrade_done_str := 'true';
                END IF;
                
            EXCEPTION
                WHEN OTHERS THEN
                    log_error('Error occured during the cursor operation, closing cursor due to ' || SQLERRM, log_file);
                    CLOSE rej_cursor;
                    ROLLBACK;
                    insert_error_log(thread, 'adapter_migrate.upgrade_rejected_messages',2, -20030, 'Error occured during cursor operation # ' || SQLCODE || '-' ||  SQLERRM);
                    upgrade_done := false;
                    upgrade_done_str := 'false';
            END;
        EXCEPTION
            WHEN OTHERS THEN
            log_error('Error occured while invoking upgrade_rejected_message due to ' || SQLERRM, log_file);
            ROLLBACK;
            upgrade_done_str := 'false';
            insert_error_log(thread, 'adapter_migrate.upgrade_rejected_messages',2, -20031, 'Error occured while invoking upgrade_rejected_messages # ' || SQLCODE || '-' || SQLERRM);
	END;
        close_logfile(log_file);
       -- num_rej_part_count := get_rejected_messages_count(rejected_message_count_query);
        --insert_error_log(thread, 'adapter_migrate.upgrade_rejected_messages',0, -20032, 'Final count  = ' || num_rej_part_count || ' upgraded = ' || upgrade_done_str);
        return upgrade_done;
    END upgrade_rejected_messages;

    
    FUNCTION get_rejected_messages_count(query IN VARCHAR2) return NUMBER 
    IS
        num_rejected_messages NUMBER := 0;
        BEGIN
            EXECUTE IMMEDIATE query into num_rejected_messages;
            RETURN num_rejected_messages;
    END get_rejected_messages_count;

    FUNCTION upgrade_rejected_message(rejected_message IN REJECTED_MESSAGE%ROWTYPE,thread INTEGER,  log_file IN utl_file.file_type) return integer 
    IS
    srm_ids varchar(100) := 'SCA_REJECTED_MSG_IDS';
    flow_ids varchar(100) := 'FLOW_INSTANCE_IDS';
    fault_ids varchar(100) := 'FAULT_IDS';
    composite_instance_ids varchar(100) := 'COMPOSITE_INSTANCE_IDS';
    srm_id number;  -- rejected message sequence number
    iflow_id number;
    composite_dn composite_dn_type;
    composite_partition_id integer;
    composite_sca_entity_id integer;
    service_sca_entity_id integer;
    fault_id number;
    fault_count integer;
    flow_count integer;
    ret_fault_id number;
    ret_flow_id number;
    composite_id number;
    retr_count number := 0;
    sca_entity_obj sca_entity_type;
    BEGIN
    log_debug('upgrade_rejected_message called ' || rejected_message.ID, log_file);                    
    --Parse composite_dn into individual parts e.g. partition id, composite name, version
    composite_dn := composite_dn_type(rejected_message.COMPOSITE_DN);
    log_debug('Looking for composite_partition_id for ' || composite_dn.partition_name, log_file);
        BEGIN
            cache_util.get_cache_query_type_4_output(name => composite_dn.partition_name, sca_partition_id => composite_partition_id);
            log_debug('Found ' || composite_partition_id || ' for ' || composite_dn.partition_name, log_file);
        EXCEPTION
            WHEN NO_DATA_FOUND THEN
               log_error('Unable to find partition_id for ' || composite_dn.partition_name, log_file);
               --insert_error_log(thread,'upgrade_rejected_message',2, -20009, 'Unable to find partition_id for ' || composite_dn.partition_name || ' for ' || rejected_message.ID || '  # ' ||SQLCODE || '-' || SQLERRM);
               --return WARNING if partition does not exist
               return 1;
            WHEN OTHERS THEN
               log_error('Error occured while trying to find partition_id for ' || composite_dn.partition_name || ' due to ' || SQLERRM, log_file);
               insert_error_log(thread,'upgrade_rejected_message' ,2, -20010, 'Error occured while trying to find partition_id for ' || composite_dn.partition_name 
                                        || ' for ' || rejected_message.ID || ' # ' ||SQLCODE || '-' || SQLERRM);
               return 2;
        END;
        if composite_partition_id > 0 then
            log_debug('partition id = '||  composite_partition_id, log_file);
            log_debug('composite name  = '||  composite_dn.composite_name, log_file);
            log_debug('revision  = '||  composite_dn.revision, log_file);
            BEGIN
                select id into composite_sca_entity_id from sca_entity where sca_partition_id = composite_partition_id and 
                   composite = composite_dn.composite_name and revision = composite_dn.revision --and label = composite_dn.label 
                   and type='composite' AND STATE = 'active' and rownum = 1;
            EXCEPTION
                WHEN NO_DATA_FOUND THEN
                    log_error('Did not find composite_sca_entity_id for composite :' || composite_dn.composite_name || ', revision :' || composite_dn.revision || ', partition :' ||  composite_partition_id  || ', label :' || composite_dn.label, log_file);
                    --insert_error_log(thread,'upgrade_rejected_message',2, -20011, 'Did not find composite_sca_entity_id for composite : ' || composite_dn.composite_name || ' revision : ' || composite_dn.revision || ', partition : ' || composite_partition_id || ' for ' || rejected_message.ID || ' # ' || SQLCODE || '-' || SQLERRM);
                     
                    -- At this point return WARNING, it seems like this particular record is pertaining
                    -- to a composite that has been purged or in-active, it does not make sense
                    -- to create a recoverable fault at this point
                    return 1; --return WARNING
                WHEN OTHERS THEN
                    log_error('Error occured while trying to find composite_sca_entity_id for ' || composite_dn.composite_name || ' due to ' || SQLERRM, log_file);
                    insert_error_log(thread, 'upgrade_rejected_message',2, -20012, 'Error occured while trying to find composite_sca_entity_id for Composite ' 
                                              || composite_dn.composite_name  || ',  revision :' || composite_dn.revision || ', partition :' ||  composite_partition_id  
                                              || ', label :' || composite_dn.label ||  ' for rejected message : ' ||  rejected_message.ID || ' # ' || SQLCODE || '-' || SQLERRM);
                    return 2; --return ERROR
            END;
                      
            BEGIN
                select id into service_sca_entity_id from sca_entity where sca_partition_id = composite_partition_id and 
                    composite = composite_dn.composite_name and  name = rejected_message.service_name and  revision = composite_dn.revision and type='service' AND STATE = 'active' and rownum = 1;
            EXCEPTION
                WHEN NO_DATA_FOUND THEN
                    log_error('Did not find sca_entity_id for composite:' || composite_dn.composite_name ||  ', service :' || rejected_message.service_name || ', partition :' || composite_partition_id || ', revision :' || composite_dn.revision || ', this is an error ', log_file);
                    --insert_error_log(thread,'upgrade_rejected_message', 0, -20013, 'Did not find sca_entity_id for composite : ' || composite_dn.composite_name ||
                    --                    ', service : ' || rejected_message.service_name || ' for ' || rejected_message.ID || ' # ' || SQLCODE || '-' || SQLERRM);
                    --sca_entity_obj := sca_entity_type(null, composite_partition_id, composite_dn.composite_name,
                    --    composite_dn.revision, composite_dn.label, rejected_message.service_name, 'service',
                    --    null, 'active', null);
                    --service_sca_entity_id := insert_sca_entity(sca_entity_obj);
                    --dumpScaEntities(log_file);
                    --if service_sca_entity_id <= 0 then
                    --  log_error('Unable to create sca_entity for composite:' || composite_dn.composite_name ||  ', service :' || rejected_message.service_name, log_file);
                    insert_error_log(thread,'upgrade_rejected_message', 2, -200014 , 'Unable to find sca_entity for composite :' || composite_dn.composite_name 
                                             ||  ', service : ' || rejected_message.service_name || ' for ' || rejected_message.ID);
                    return 2;
                WHEN OTHERS THEN
                    log_error('Error occured while trying to find sca_entity_id for ' || composite_dn.composite_name || ' due to ' || SQLERRM, log_file);
                    insert_error_log(thread, 'upgrade_rejected_message',2, -20015 , 'Error occured while trying to find sca_entity_id for ' || composite_dn.composite_name || 
                                               ', service : ' || rejected_message.service_name || ' for ' || rejected_message.ID || ' # ' || SQLCODE || '-' || SQLERRM);
                    return 2;
            END;
            log_debug('Found composite sca entity id  = '||  composite_sca_entity_id || ' service sca entity id ' || service_sca_entity_id, log_file);
        end if;     

        -- Generate Rejected Message Id, and Fault Id
        srm_id := get_next_id(srm_ids);

        -- Generate composite_instance_id
        composite_id := get_next_id(composite_instance_ids);

        fault_id :=  get_next_id(fault_ids);
        iflow_id := get_next_id(flow_ids);
        
   
        log_debug('Final fault_id is ' || fault_id || ' and flow_id ' || iflow_id, log_file);
        --insert_error_log(thread, 'upgrade_rejected_message',0, -20018, 'Final fault_id : ' || fault_id || ', flow_id : ' || iflow_id || ', composite_id : ' || composite_id ||
        --         ' for ' ||  rejected_message.ID);

        -- Create a Flow Instance if one did not exist
        BEGIN
            INSERT INTO SCA_FLOW_INSTANCE (
                FLOW_ID,
                ECID,
                CONVERSATION_ID,
                COMPOSITE_SCA_ENTITY_ID,
                SCA_PARTITION_ID,
                UNHANDLED_FAULTS,
                UPDATED_TIME
            )
            VALUES (
                iflow_id,
                rejected_message.ECID,
                rejected_message.PROTOCOL_MESSAGE_ID,
                composite_sca_entity_id,
                composite_partition_id, 
                1 ,
                systimestamp
            );
            log_debug(' Inserted into SCA_FLOW_INSTANCE ', log_file);
                -- insert_error_log(thread, 'upgrade_rejected_message',0, -20019, 'Inserted into SCA_FLOW_INSTANCE with flow_id : ' || iflow_id || ' for ' ||
                -- rejected_message.ID);
        EXCEPTION
               WHEN OTHERS THEN
                   log_error('Unable to insert into SCA_FLOW_INSTANCE for ' || rejected_message.ID || ' due to ' || SQLERRM || ', the script will continue', log_file);
                   insert_error_log(thread, 'upgrade_rejected_message',2, -20020, 'Unable to insert into SCA_FLOW_INSTANCE flow_id : ' || iflow_id || 
                                            ' for ' ||  rejected_message.ID || ' # ' || SQLCODE || '-' || SQLERRM);
                   return 2;
        END;

           BEGIN
               INSERT INTO SCA_FLOW_TO_CPST (
                    COMPOSITE_ID,
                    FLOW_ID,
                    COMPOSITE_SCA_ENTITY_ID,
                    CREATED_TIME, 
                    UPDATED_TIME,
                    ACTIVE_COMPONENT_INSTANCES,
                    SCA_PARTITION_ID
               )
               VALUES (
                    composite_id,
                    iflow_id,
                    composite_sca_entity_id,
                    rejected_message.CREATED_TIME,
                    rejected_message.CREATED_TIME,
                    0,
                    composite_partition_id
               );
               log_debug(' Inserted into SCA_FLOW_TO_CPST ', log_file);
               --insert_error_log(thread, 'upgrade_rejected_message',0, -20021, 'Inserted into SCA_FLOW_TO_CPST COMPOSITE_ID : ' || composite_id || ',FLOW_ID : ' ||
               --                          iflow_id || ' for ' ||  rejected_message.ID);
           EXCEPTION
               WHEN OTHERS THEN
                   log_error('Unable to insert into SCA_FLOW_TO_CPDT for ' || rejected_message.ID || ' due to ' || SQLERRM || ', the script will continue', log_file);
                   insert_error_log(thread, 'upgrade_rejected_message',2, -20022, 'Unable to insert into SCA_FLOW_TO_CPST COMPOSITE_ID : ' || composite_id ||  
                                            ',FLOW_ID : ' || iflow_id || ' for ' ||  rejected_message.ID || ' # ' || SQLCODE || '-' || SQLERRM);
                   return 2;
           END;
       

       retr_count := nvl(rejected_message.RETRY_COUNT, 0);

       --Create a Common Fault if one did not exist
       
           BEGIN
               INSERT INTO SCA_COMMON_FAULT(
                                    ID,
                                    RETRY_COUNT,
                                    FLOW_ID,
                                    COMPONENT_TYPE,
                                    SCA_ENTITY_ID,
                                    COMPOSITE_SCA_ENTITY_ID, 
				    OWNER_SCA_ENTITY_ID,
                                    OWNER_COMPONENT_TYPE,
                                    OWNER_COMPOSITE_SCA_ENTITY_ID,
                                    FAULT_TYPE,
                                    FAULT_NAME,
                                    ERROR_MESSAGE,
                                    EXCEPTION_TRACE,
                                    STATE ,
                                    SCA_PARTITION_ID,
                                    OWNER_SCA_PARTITION_ID,
                                    COMPOSITE_ID,
                                    CREATION_DATE,
                                    MODIFY_DATE)
                                    VALUES (
                                     fault_id,
                                     retr_count,
                                     iflow_id,
                                     'REJ_MSG',
                                     service_sca_entity_id,
                                     composite_sca_entity_id,
                                     service_sca_entity_id,
                                     'REJ_MSG',
                                     composite_sca_entity_id,
                                     'SYSTEM',
                                     '{http://schemas.oracle.com/sca/rejectedmessages}' || rejected_message.service_name,
                                     rejected_message.ERROR_MESSAGE,
                                     rejected_message.STACK_TRACE,
                                     256,
                                     composite_partition_id,
                                     composite_partition_id,
                                     composite_id,
                                     rejected_message.CREATED_TIME,
                                     rejected_message.CREATED_TIME
                                   );
                              log_debug(' Inserted into SCA_COMMON_FAULT', log_file);
                       -- insert_error_log(thread, 'upgrade_rejected_message',0, -20023, 'Inserted into SCA_COMMON_FAULT fauld_id : ' || fault_id || ',flow_id :' || 
                       --                          iflow_id ||' for ' ||  rejected_message.ID);
            EXCEPTION
               WHEN OTHERS THEN
                   log_error('Unable to insert into SCA_COMMON_FAULT for ' || rejected_message.ID || ' due to ' || SQLERRM || ', the script will continue', log_file);
                   insert_error_log(thread, 'upgrade_rejected_message',2, -20024, 'Unable to insert into SCA_COMMON_FAULT with input fault_id : ' || fault_id || 
                                             ' for ' ||  rejected_message.ID || ' # ' || SQLCODE || '-' || SQLERRM);
                   return 2;
            END;
       
                            

        BEGIN
             insert_rejected_message(rejected_message.ECID, srm_id, iflow_id, rejected_message.BATCH_ID, rejected_message.CREATED_TIME,
                            rejected_message.SERVICE_NAME, rejected_message.BINDING_TYPE, rejected_message.PROTOCOL_MESSAGE_ID,
                            rejected_message.PROTOCOL_CORRELATION_ID, rejected_message.ERROR_CATEGORY,rejected_message.ERROR_CODE,
                            rejected_message.ERROR_MESSAGE, rejected_message.STACK_TRACE, retr_count,
                            rejected_message.MESSAGE_ORIGIN_REFERENCE,rejected_message.ADDITIONAL_PROPERTIES, rejected_message.TENANT_ID,
                            composite_partition_id);
             log_debug('Inserted into SCA_REJECTED_MESSAGE :' ||
                           ' ecid :' || rejected_message.ECID || ' id :' || rejected_message.ID ||
                                        ' flow id :' || iflow_id , log_file);
        EXCEPTION
            WHEN OTHERS THEN
                log_error('Unable to insert into SCA_REJECTED_MESSAGE :' ||
                           ' ecid :' || rejected_message.ECID || ' id :' || rejected_message.ID ||
                           ' flow id :' || iflow_id  || ' due to ' || SQLERRM, log_file);
                   insert_error_log(thread, 'upgrade_rejected_message',2, -20025, 'Unable to insert into SCA_REJECTED_MESSAGE for ' ||  rejected_message.ID ||
                                           ' # ' || SQLCODE || '-' || SQLERRM);
                return 2;
        END;
        BEGIN 
            update_rejected_message(rejected_message.ID, iflow_id);
        EXCEPTION
            WHEN OTHERS THEN
              log_error('Unable to update REJECTED_MESSAGE for ' || rejected_message.ID || ' due to ' || SQLERRM || ', the script will continue', log_file);
              insert_error_log(thread, 'upgrade_rejected_message',2, -20026, 'Unable to updated REJECTED_MESSAGE for ' ||  rejected_message.ID ||  ',flow_id : ' ||
                                                iflow_id ||  ' # ' || SQLCODE || '-' || SQLERRM);
              return 2;
        END;
        return 0;
    END;
    
    PROCEDURE insert_rejected_message
    (
        ecid IN VARCHAR2, 
        srm_id IN NUMBER, 
        flow_id  IN NUMBER, 
        batch_id IN VARCHAR2,
        created_time IN timestamp,
        service_name IN VARCHAR2,
        binding_type IN VARCHAR2,
        protocol_message_id IN VARCHAR2, 
        protocol_correlation_id IN VARCHAR2,
        error_category IN VARCHAR2, 
        error_code IN VARCHAR2,
        error_message IN CLOB, 
        stack_trace IN CLOB,
        retry_count IN NUMBER,
        message_origin_reference IN VARCHAR2, 
        additional_properties IN VARCHAR2,
        tenant_id IN NUMBER,
        partition_id IN NUMBER
    ) 
    IS 
    BEGIN
       INSERT INTO SCA_REJECTED_MESSAGE
                                       (
                                           ECID,
                                           SRM_ID,
                                           FLOW_ID,
                                           BATCH_ID,
                                           CREATED_TIME,
                                           SERVICE_NAME,
                                           BINDING_TYPE,
                                           PROTOCOL_MESSAGE_ID,
                                           PROTOCOL_CORRELATION_ID,
                                           ERROR_CATEGORY,
                                           ERROR_CODE,
                                           ERROR_MESSAGE,
                                           STACK_TRACE,
                                           RETRY_COUNT,
                                           MESSAGE_ORIGIN_REFERENCE,
                                           ADDITIONAL_PROPERTIES,
                                           TENANT_ID,
                                           SCA_PARTITION_ID,
                                           UPGRADED
                                        )
                                        VALUES
                                        (
                                           ecid,
                                           srm_id,
                                           flow_id,
                                           batch_id,
                                           created_time,
                                           service_name,
                                           binding_type,
                                           protocol_message_id,
                                           protocol_correlation_id,
                                           error_category,
                                           error_code,
                                           error_message,
                                           stack_trace,
                                           retry_count,
                                           message_origin_reference, 
                                           additional_properties,
                                           tenant_id,
                                           partition_id,
                                           1
                                         );
    END;
    PROCEDURE update_rejected_message
    (
        in_id      IN NUMBER,
        in_flow_id IN NUMBER
    )
    IS 
    BEGIN
        UPDATE REJECTED_MESSAGE SET FLOW_ID = in_flow_id where ID = in_id;
    END;
    

    PROCEDURE mark_rejected_message_error
    (
        in_id      IN NUMBER,
        ecid       IN VARCHAR,
        in_flowid  IN NUMBER,
        log_file IN utl_file.file_type         
    )
    IS 
    BEGIN
        log_error('Error occured while upgrading rejected message ' || in_id, log_file);
        update_rejected_message(in_id, in_flowid);
    EXCEPTION
        WHEN OTHERS THEN 
            log_error('Unable to update faulted rejected message ' || in_id || ' ignoring this error', log_file);
    END;

END adapter_migrate;
/
show errors;
