create or replace type edn_target_type as object
(
    edn_target varchar2(1253),
    partition_name varchar2(200),
    composite_name varchar2(500),
    revision varchar2(50),
    label varchar2(500),
    subscriber_name varchar2(500),
    constructor function edn_target_type(p_edn_target varchar2) return self as result,
    member function get_partition_name return varchar2,
    member function get_composite_name return varchar2,
    member function get_version_name return varchar2,
    member function get_label_name return varchar2,
    member function get_subscriber_name return varchar2
)
/
create or replace type body edn_target_type  as

  constructor function edn_target_type(p_edn_target varchar2) return self as result is
  pos1 integer;
  pos2 integer;
  pos3 integer;
  begin
     SELF.edn_target := p_edn_target;
     pos1 := INSTR(edn_target,'/');
     SELF.partition_name := SUBSTR(edn_target,1,pos1-1);
     pos2 := INSTR(edn_target,'!',pos1+1);
     SELF.composite_name := SUBSTR(edn_target,pos1+1,pos2-pos1 -1);
     pos1 := INSTR(edn_target,'*' ,pos2+1);
     pos3 := INSTR(edn_target,'/' ,pos1+1);
     SELF.revision := SUBSTR(edn_target,pos2+1,pos1-pos2-1);
     SELF.label := SUBSTR(edn_target,pos1+1, pos3-pos1-1);  	 
     SELF.subscriber_name := SUBSTR(edn_target, pos3+1);
     return;
  end edn_target_type;
  member function get_composite_name return varchar2 is
  begin
     return composite_name;
  end get_composite_name;
  member function get_partition_name return varchar2 is
  begin
     return partition_name;
  end get_partition_name;
  member function get_version_name return varchar2 is
  begin
     return revision;
  end get_version_name;
  member function get_label_name return varchar2 is
  begin
     return label;
  end get_label_name;  
  member function get_subscriber_name return varchar2 is
  begin
     return subscriber_name;
  end get_subscriber_name;  
 end; 
/

create or replace function clob2Blob(p_clob clob) return blob is
      l_blob         blob;
      l_dest_offsset integer := 1;
      l_src_offsset  integer := 1;
      l_lang_context integer := dbms_lob.default_lang_ctx;
      l_warning      integer;
   begin

      if p_clob is null then
         return null;
      end if;

      dbms_lob.createTemporary(lob_loc => l_blob
                              ,cache   => false);
      dbms_lob.converttoblob(dest_lob     => l_blob
                            ,src_clob     => p_clob
                            ,amount       => DBMS_LOB.LOBMAXSIZE
                            ,dest_offset  => l_dest_offsset
                            ,src_offset   => l_src_offsset
                            ,blob_csid    => 0
                            ,lang_context => l_lang_context
                            ,warning      => l_warning);

      return l_blob;
end;
/


create or replace 
PACKAGE edn_migrate AS

    FUNCTION upgrade_edn_messages(batch_size in number default 10000, stoptime  in date default null, max_count number default 0, 
                                        thread in integer default -1, dop in integer default 1) RETURN boolean ;
    FUNCTION upgrade_edn_message(thread in integer, edn_message IN EDN_EVENT_ERROR_STORE%ROWTYPE, log_file IN utl_file.file_type) return integer;

    FUNCTION get_edn_messages_count(query in varchar2) RETURN NUMBER;

    PROCEDURE insert_rejected_message
    (
        srm_id IN NUMBER, 
        iflow_id  IN NUMBER, 
        message_origin_reference VARCHAR2,
        created_time IN timestamp,
        error_message CLOB,
        stack_trace IN CLOB,
        retry_count IN NUMBER,
        namespace IN VARCHAR2, 
        local_name IN VARCHAR2,
        srm_edn_target IN VARCHAR2,
        srm_state IN NUMBER,
        msg_encoding IN VARCHAR2,
        msg_payload IN BLOB,
        payload_type IN NUMBER,
        tracking_context_properties IN BLOB,
        resubmit_op IN VARCHAR2,
        partition_id IN NUMBER
    );

    PROCEDURE update_edn_error_store
    (
        in_id      IN RAW,
        in_flow_id IN NUMBER
    );

    PROCEDURE mark_edn_message_error
    (
        in_id      IN RAW,
        in_flow    IN NUMBER,
        log_file IN utl_file.file_type
    );

END edn_migrate;
/

show errors

   
create or replace PACKAGE BODY edn_migrate AS


    FUNCTION upgrade_edn_messages(batch_size in number default 10000, stoptime  in date default null, max_count number default 0, 
                                        thread in integer default -1, dop in integer default 1) RETURN boolean 
    IS
    TYPE EDN_ERROR_TYPE IS TABLE OF EDN_EVENT_ERROR_STORE%ROWTYPE;
    EDN_ERROR_ARRAY EDN_ERROR_TYPE;
    log_file utl_file.file_type;
    limit_val number :=  1000;
    ret_value integer := 0;
    edn_message_count number := 0;
    edn_cursor sys_refcursor;
    edn_message_query varchar2(500);
    edn_message_count_query varchar2(500);
    main_query varchar2(75) := 'SELECT * from EDN_EVENT_ERROR_STORE WHERE FLOW_ID IS NULL AND STATE=1 ';
    main_count_query varchar2(95) := 'SELECT COUNT(1) from EDN_EVENT_ERROR_STORE WHERE FLOW_ID IS NULL AND STATE=1 and ROWNUM=1 ';
    order_by_time_query varchar2(30) := ' ORDER BY CREATION_TIME DESC ';
    rownum_query varchar2(20) := ' and rownum <= ';
    parallel_query varchar2(200) :=' and mod (dbms_utility.get_hash_value(MSG_ID,0,' || dop || '),' || dop || ')=' ||thread ;
    upgrade_done boolean := true;
    upgrade_done_str varchar2(10) := 'true';
    commit_size number :=  100;
    commit_size_count number;
    num_rej_part_count number;
    BEGIN
        BEGIN
            dbms_output.enable(NULL);
            log_file := get_logfile();
            dumpScaEntities(log_file); --dump it at the begining
            log_debug('edn_migrate.upgrade_edn_messages start', log_file);
            select count(1) into edn_message_count from EDN_EVENT_ERROR_STORE WHERE FLOW_ID IS NULL AND STATE=1;
            log_error('edn_migrate.upgrade_edn_messages number of records to be processed in this batch = ' || edn_message_count, log_file);
            --insert_error_log(thread, 'edn_migrate.upgrade_edn_messages',0, -30001, 'Number of records to be processed in this batch = ' || edn_message_count);
            IF max_count = 0 THEN
                edn_message_query := main_query;
                edn_message_count_query := main_count_query;
            ELSE
                edn_message_query := main_query ||rownum_query || max_count ;
                edn_message_count_query := main_count_query  ;
            END IF;
            if thread >=0 then
                edn_message_query := edn_message_query || parallel_query ;
                edn_message_count_query := edn_message_count_query || parallel_query ;
            end if;
            edn_message_query := edn_message_query || order_by_time_query;
            log_debug('upgrade_edn_messages query = ' ||edn_message_query,log_file);
            --insert_error_log(thread, 'edn_migrate.upgrade_edn_messages',0, -30002, 'EDN Message Query = ' || edn_message_query);
           -- num_rej_part_count := get_edn_messages_count(edn_message_count_query);
            --insert_error_log(thread, 'edn_migrate.upgrade_edn_messages',0, -30003, 'EDN Rejected Message COUNT Query = ' || edn_message_count_query || 'Number of records to process by this instance =>' || num_rej_part_count);
            BEGIN
                log_debug('Opening cursor for edn messages', log_file);
                OPEN edn_cursor FOR edn_message_query;
                LOOP
                    FETCH edn_cursor BULK COLLECT INTO EDN_ERROR_ARRAY LIMIT batch_size;
                    IF EDN_ERROR_ARRAY.COUNT = 0 THEN
                        log_debug('No records found in edn_event_error_store', log_file);
                        CLOSE edn_cursor;
                        close_logfile(log_file);
                        COMMIT;
                        --insert_error_log(thread, 'edn_migrate.upgrade_edn_messages',0, -30004, 'Cursor fetch returned 0, returning');
                        num_rej_part_count := get_edn_messages_count(edn_message_count_query);
                        IF (num_rej_part_count > 0) THEN 
                            RETURN false;
                        ELSE 
                            RETURN true;
                        END IF;
                    END IF;
                    IF (sysdate >= stoptime) THEN
                        log_debug('Stop time reached. EDN messages still left for upgrade',log_file);
                        CLOSE edn_cursor;
                        close_logfile(log_file);
                        COMMIT;
                        --insert_error_log(thread, 'edn_migrate.upgrade_edn_messages',0, -30005, 'Stop time reached, exiting...');
                        return false;
                    END IF;
                    IF is_stop_injected THEN
                        log_debug('Stop injected. EDN messages still left for upgrade',log_file);
                        CLOSE edn_cursor;
                        close_logfile(log_file);
                        COMMIT;
                        --insert_error_log(thread, 'edn_migrate.upgrade_edn_messages',0, -30006, 'Stop injected, exiting...');
                        return false;
                    END IF;
                    commit_size_count := 1;
                    FOR I in 1..EDN_ERROR_ARRAY.COUNT
                    LOOP
                        BEGIN
                            IF (sysdate >= stoptime) THEN
                                log_debug('Stop time reached. EDN messages still left for upgrade',log_file);
                                CLOSE edn_cursor;
                                close_logfile(log_file);
                                COMMIT;
                                --insert_error_log(thread, 'edn_migrate.upgrade_edn_messages',0, -30007, 'Stop time reached while looping through cursor, exiting...');
                                return false;
                            END IF;
                            IF is_stop_injected THEN
                                log_debug('Stop injected. Rejected message records still left for upgrade',log_file);
                                CLOSE edn_cursor;
                                close_logfile(log_file);
                                COMMIT;
                                --insert_error_log(thread, 'edn_migrate.upgrade_edn_messages',0, -30008, 'Stop injected while looping through cursor, exiting...');
                                return false;
                            END IF;
                            ret_value := upgrade_edn_message(thread, EDN_ERROR_ARRAY(I), log_file);
                            --ERRORs are marked as -1
                            IF ret_value = 2 THEN -- ERROR
                                mark_edn_message_error(EDN_ERROR_ARRAY(I).MSG_ID,-1, log_file);
                            END IF;
                            --WARNINGs are marked as -9 <for now ;)>
                            IF ret_value = 1 THEN -- WARNING
                                mark_edn_message_error(EDN_ERROR_ARRAY(I).MSG_ID,-9, log_file);
                            END IF;
                            IF commit_size_count >= commit_size  THEN
                               COMMIT; 
                               commit_size_count := 1;
                            ELSE
                               commit_size_count := commit_size_count + 1;
                            END IF;
                        EXCEPTION
                            WHEN OTHERS THEN
                                insert_error_log(thread, 'edn_migrate.upgrade_edn_messages',2, -30027, 'Unable to process single edn message, ' ||
                                                           EDN_ERROR_ARRAY(I).MSG_ID || '  # ' || SQLCODE || '-' || SQLERRM);
                                mark_edn_message_error(EDN_ERROR_ARRAY(I).MSG_ID,-1, log_file);
                                COMMIT;
                        END;
                    END LOOP;
                    log_debug('Comitting after ' || batch_size || ' records', log_file);
                    COMMIT;
                END LOOP;
                log_debug('Closing cursor for edn messages', log_file);
                CLOSE edn_cursor;
                COMMIT; -- Final commit
                num_rej_part_count := get_edn_messages_count(edn_message_count_query);
                IF (num_rej_part_count > 0) THEN 
                    upgrade_done := false;
                    upgrade_done_str := 'false';
                ELSE 
                    upgrade_done := true;
                    upgrade_done_str := 'true';
                END IF;
                
                --insert_error_log(thread, 'edn_migrate.upgrade_rejected_messages',0, -30028, 'Final commit done');
            EXCEPTION 
                WHEN OTHERS THEN
                    log_error('Error occured during the cursor operation, closing cursor due to ' || SQLERRM, log_file);
                    CLOSE edn_cursor;
                    ROLLBACK;
                    insert_error_log(thread, 'edn_migrate.upgrade_edn_messages',2, -30029, 'Error occured during cursor operation # ' || SQLCODE || '-' ||  SQLERRM);
                    upgrade_done := false;
                    upgrade_done_str := 'false';
            END;
        EXCEPTION 
            WHEN OTHERS THEN
                log_error('Error occured while invoking upgrade_edn_message due to ' || SQLERRM, log_file);
                CLOSE edn_cursor;
                ROLLBACK;
                insert_error_log(thread, 'edn_migrate.upgrade_edn_messages',2, -30030, 'Error occured while invoking upgrade_edn_messages # ' || SQLCODE || '-' || SQLERRM);
                upgrade_done := false;
                upgrade_done_str := 'false';
        END;
        close_logfile(log_file);
      --  num_rej_part_count := get_edn_messages_count(edn_message_count_query);
        --insert_error_log(thread, 'edn_migrate.upgrade_edn_messages',0, -30031, 'Final check before exit, count  = ' || num_rej_part_count || ', upgraded  = ' ||                                  upgrade_done_str);
        return upgrade_done;
    END upgrade_edn_messages;

    FUNCTION get_edn_messages_count(query IN VARCHAR2) return NUMBER 
    IS
        num_rejected_messages NUMBER;
        BEGIN
            EXECUTE IMMEDIATE query into num_rejected_messages;
            RETURN num_rejected_messages;

    END get_edn_messages_count;

    FUNCTION upgrade_edn_message(thread IN INTEGER, edn_message IN EDN_EVENT_ERROR_STORE%ROWTYPE, log_file IN utl_file.file_type) return integer 
    IS
    srm_ids varchar(100) := 'SCA_REJECTED_MSG_IDS';
    flow_ids varchar(100) := 'FLOW_INSTANCE_IDS';
    fault_ids varchar(100) := 'FAULT_IDS';
    composite_instance_ids varchar(100) := 'COMPOSITE_INSTANCE_IDS';
    counter integer := 0;  -- used to batch commits e.g. commit every 1000 iterations
    srm_id number;  -- rejected message sequence number
    iflow_id number;
    target edn_target_type;
    composite_partition_id integer;
    composite_sca_entity_id integer;
    subscriber_sca_entity_id integer;
    subscriber_label varchar2(500);
    upgraded_target varchar2(500);
    fault_id number;
    fault_count integer;
    ret_fault_id number;
    ret_flow_id number;
    sca_entity_obj sca_entity_type;
    composite_id number;

    BEGIN
    log_debug('upgrade_edn_message called ' || edn_message.MSG_ID, log_file);                    
    --Parse target into individual parts e.g. partition id, composite name, version, subscriber
    target := edn_target_type(edn_message.TARGET);
    log_debug('Looking for composite_partition_id for ' || target.partition_name, log_file);
        BEGIN
           cache_util.get_cache_query_type_4_output(name => target.partition_name, sca_partition_id => composite_partition_id);
           log_debug('Found ' || composite_partition_id || ' for ' || target.partition_name, log_file);
        EXCEPTION
            WHEN NO_DATA_FOUND THEN
                log_error('Unable to find composite_partition_id for ' || target.partition_name, log_file);
                --insert_error_log(thread,'upgrade_edn_message',2, -30009, 'Unable to find composite_partition_id for ' || target.partition_name || ' for ' || edn_message.MSG_ID || ' # ' ||SQLCODE || '-' || SQLERRM);
                --If partition does not exist, return warning instead, the partition
                -- may have been deleted and hence it is no longer required to process this record
                return 1; -- return WARNING
            WHEN OTHERS THEN
                log_error('Error occured while trying to find partition id for ' || target.partition_name || ' due to ' || SQLERRM, log_file);
                insert_error_log(thread,'upgrade_edn_message' ,2, -30010, 'Error occured while trying to find partition_id for ' || target.partition_name 
                                        || ' for ' || edn_message.MSG_ID || ' # ' ||SQLCODE || '-' || SQLERRM);
                return 2; -- return ERROR
        END;
        if composite_partition_id > 0 then
           log_debug('partition id = '||  composite_partition_id, log_file);
           log_debug('composite name  = '||  target.composite_name, log_file);
           log_debug('revision  = '||  target.revision, log_file);
           log_debug('subscriber name   = ['||  target.subscriber_name || ']', log_file);
           log_debug('label = '||  target.label);
           BEGIN
              select id into composite_sca_entity_id from sca_entity where sca_partition_id = composite_partition_id and 
                    composite = target.composite_name and revision = target.revision --and label = target.label 
                    and type='composite' AND STATE = 'active' and rownum = 1;
           EXCEPTION
               WHEN NO_DATA_FOUND THEN
                  log_debug('Did not find sca_entity id for composite:' || target.composite_name ||  ', partition :' || composite_partition_id || ', revision :' || target.revision || ', label =' || target.label || ' due to ' || SQLERRM || ', this record will be ignored as the corresponding composite is not in use', log_file);
                  --insert_error_log(thread,'upgrade_edn_message', 0, -30011, 'Did not find sca_entity id for composite : ' || target.composite_name ||  ',subscriber : ' ||
                  --     target.subscriber_name || ' for ' || edn_message.MSG_ID || ' # ' || SQLCODE || '-' || SQLERRM);
                  -- sca_entity_obj := sca_entity_type(null, composite_partition_id, target.composite_name,
                  --       target.revision, target.label, null, 'composite', null, 'active', null);
                  -- composite_sca_entity_id := insert_sca_entity(sca_entity_obj) ;
                  -- dumpScaEntities(log_file);
                  -- if composite_sca_entity_id <= 0 then
                  --     log_error('Unable to create sca_entity for composite:' || target.composite_name, log_file);
                  --     insert_error_log(thread,'upgrade_edn_message', 2, -300012 , 'Unable to create sca_entity for composite : ' || target.composite_name ||  ', partition :' || composite_partition_id || ', revision :' || target.revision || ', label =' || target.label ||   ',subscriber : ' || target.subscriber_name || ' for ' || edn_message.MSG_ID);
                  -- else 
                  --    log_debug('Created sca_entity for composite with id ' || composite_sca_entity_id, log_file);
                  -- end if;
                  -- At this point return WARNING, it seems like this particular record is pertaining
                  -- to a composite that has been purged or in-active, it does not make sense
                  -- to create a recoverable fault at this point
                  return 1; --return WARNING
               WHEN OTHERS THEN
                  log_error('Error occured while trying to find composite_sca_entity_id for ' || target.composite_name || ' due to ' || SQLERRM, log_file);
                  insert_error_log(thread,'upgrade_edn_message', 2, -300013 , 'Error occured while trying to find composite_sca_entity_id for composite : ' ||
                                           target.composite_name  ||  ', partition :' || composite_partition_id || ', revision :' || target.revision 
                                          || ', label =' || target.label  ||  ', subscriber : ' || target.subscriber_name || ' for ' ||
                                            edn_message.MSG_ID  || ' # ' || SQLCODE || '-' || SQLERRM);
                  return 2; -- return ERROR
           END;
        end if;
                      
                       
           BEGIN
               select id, label into subscriber_sca_entity_id, subscriber_label from sca_entity where sca_partition_id = composite_partition_id and 
                   composite = target.composite_name and revision = target.revision --and label = target.label 
                   and name = target.subscriber_name and type='component' AND 
                   STATE = 'active';
                    /*upgraded_target := target.partition_name || '/' || target.composite_name || '!' || target.revision || '*' || subscriber_label || '/' ||  
                               target.subscriber_name;*/
                   upgraded_target := edn_message.TARGET;
                   log_debug('Updated target ' || upgraded_target, log_file);
           EXCEPTION
               WHEN NO_DATA_FOUND THEN
                   log_debug('Did not find component subscriber for composite:' || target.composite_name ||  ', subscriber :' || target.subscriber_name || ', partition :' || composite_partition_id || ', revision :' || target.revision || ', label =' || target.label  || ', this record will be ignored', log_file);
                    -- insert_error_log(thread, 'upgrade_edn_message' ,0,-30014, 'Did not find component subscriber for composite : ' || target.composite_name ||
                    --                         ',subscriber : ' || target.subscriber_name  ||   ' for ' || edn_message.MSG_ID  || ' # ' || SQLCODE || '-' || SQLERRM);
                    -- sca_entity_obj := sca_entity_type(null, composite_partition_id, target.composite_name,
                    --     target.revision, target.label, target.subscriber_name, 'component', null, 'active', null);
                    -- subscriber_sca_entity_id := insert_sca_entity(sca_entity_obj) ;
                    -- dumpScaEntities(log_file);
                    -- if subscriber_sca_entity_id <= 0 then
                    --     log_error('Unable to create sca_entity for subscriber:' || target.composite_name ||  ', subscriber :' || target.subscriber_name, log_file);
                    --     insert_error_log(thread, 'upgrade_edn_message', 2, -30015, 'Unable to create sca_entity for composite  : ' || target.composite_name ||  ',subscriber : ' || target.subscriber_name || ' for ' || edn_message.MSG_ID  || ' # ' || SQLCODE || '-' || SQLERRM);
                    -- else 
                    --     log_debug('Created subscriber sca entity with id ' || subscriber_sca_entity_id);
                    -- end if;
                    -- At this point return WARNING, it seems like this particular record is pertaining
                    -- to a subscriber information that has been purged or in-active, it does not make sense
                    -- to create a recoverable fault at this point
                    return 1; --return WARNING
               WHEN OTHERS THEN
                   log_error('Error occured whule trying to find subscriber sca_entity_id for ' || target.composite_name || ' due to ' || SQLERRM, log_file);
                   insert_error_log(thread, 'upgrade_edn_message', 2, -30016, 'Error occured while trying to find subscriber sca_entity_id for composite : ' ||
                                             target.composite_name || ', partition :' || composite_partition_id || ', revision :' || target.revision 
                                             || ', label =' || target.label ||  ', subscriber : ' || target.subscriber_name || ' for ' ||
                                             edn_message.MSG_ID  || ' # ' || SQLCODE || '-' || SQLERRM);
                   return 2; --return ERROR 
            END;
            log_debug('Found composite sca entity id  = '||  composite_sca_entity_id || ' subscriber sca entity id ' || subscriber_sca_entity_id, log_file);
            -- Generate Rejected Message Id,Fault Id, Flow Id
            -- There is no way in EDN to make the records in error store
            -- to pre-exising faults
            srm_id := get_next_id(srm_ids);
            fault_id :=  get_next_id(fault_ids);
            iflow_id :=  get_next_id(flow_ids);
            composite_id := get_next_id(composite_instance_ids);
            log_debug('fault_id is ' || fault_id || ', flow_id ' || iflow_id || ', composite_id ' || composite_id, log_file);
            -- insert_error_log(thread, 'upgrade_edn_message',0, -30017, 'Generated fault_id : ' || fault_id || ',flow_id : ' || iflow_id || ' for ' ||  edn_message.MSG_ID);
            BEGIN
                           INSERT INTO SCA_FLOW_INSTANCE (
                                       FLOW_ID,
                                       FLOW_CORRELATION_ID,
                                       COMPOSITE_SCA_ENTITY_ID,
                                       SCA_PARTITION_ID,
                                       ACTIVE_COMPONENT_INSTANCES,
                                       UNHANDLED_FAULTS,
                                       RECOVERABLE_FAULTS,
                                       UPDATED_TIME
                                   )
                                   VALUES (
                                       iflow_id,
                                       edn_message.MSG_ID,
                                       composite_sca_entity_id,
                                       composite_partition_id, 
                                       0,
                                       0 ,
                                       1,
                                       systimestamp
                                   );
                      log_debug(' Inserted into SCA_FLOW_INSTANCE ', log_file);
                      -- insert_error_log(thread, 'upgrade_edn_message',0, -30018, 'Inserted into SCA_FLOW_INSTANCE flow_id : ' || iflow_id || ' for ' ||  edn_message.MSG_ID);
            EXCEPTION 
                 WHEN OTHERS THEN
                     log_error('Unable to insert into SCA_FLOW_INSTANCE for ' || edn_message.MSG_ID || ' due to ' || SQLERRM || ', the script will continue', log_file);
                     insert_error_log(thread, 'upgrade_rejected_message',2, -30019, 'Unable to insert into SCA_FLOW_INSTANCE flow_id : ' || iflow_id ||
                          ' for ' ||  edn_message.MSG_ID || ' # ' || SQLCODE || '-' || SQLERRM);
                     return 2;
            END;
            BEGIN
		      INSERT INTO SCA_FLOW_TO_CPST (
                                       COMPOSITE_ID,
                                       FLOW_ID,
                                       COMPOSITE_SCA_ENTITY_ID,
                                       CREATED_TIME, 
                                       UPDATED_TIME,
                                       ACTIVE_COMPONENT_INSTANCES,
                                       SCA_PARTITION_ID
                                       )
                                       VALUES (
                                            composite_id,
                                            iflow_id,
                                            composite_sca_entity_id,
                                            edn_message.CREATION_TIME,
                                            edn_message.CREATION_TIME,
                                            0,
                                            composite_partition_id
                                       );
                      log_debug(' Inserted into SCA_FLOW_TO_CPST ', log_file);
                      -- insert_error_log(thread, 'upgrade_edn_message',0, -30020, 'Inserted into SCA_FLOW_TO_CPST COMPOSITE_ID : ' || composite_id || ',FLOW_ID : '||
                      --                          iflow_id || ' for ' ||  edn_message.MSG_ID);
            EXCEPTION
                WHEN OTHERS THEN
                      log_error('Unable to insert into SCA_FLOW_TO_CPST for ' || edn_message.MSG_ID || ' due to '  || SQLERRM || ', the script will continue', log_file);
                      insert_error_log(thread, 'upgrade_rejected_message',2, -30021, 'Unable to insert into SCA_FLOW_TO_CPST COMPOSITE_ID : ' || composite_id ||
                                               ',FLOW_ID : ' || iflow_id || ' for ' ||  edn_message.MSG_ID || ' # ' || SQLCODE || '-' || SQLERRM);
                      return 2; 
            END;

            BEGIN
                        INSERT INTO SCA_COMMON_FAULT(
                                       ID,
                                       RETRY_COUNT,
                                       FLOW_ID,
                                       COMPONENT_TYPE,
                                       SCA_ENTITY_ID,
                                       COMPOSITE_SCA_ENTITY_ID, 
                                       OWNER_SCA_ENTITY_ID,
                                       OWNER_COMPONENT_TYPE,
                                       OWNER_COMPOSITE_SCA_ENTITY_ID,
                                       FAULT_NAME,
                                       FAULT_TYPE,
                                       ERROR_MESSAGE,
                                       EXCEPTION_TRACE,
                                       STATE ,
                                       SCA_PARTITION_ID,
                                       OWNER_SCA_PARTITION_ID,
                                       COMPOSITE_ID,
                                       EVENT_NAME,
                                       SRM_NAMESPACE,
                                       CREATION_DATE,
                                       MODIFY_DATE)
                                    VALUES (
                                         fault_id,
                                         0,
                                         iflow_id,
                                         'REJ_MSG',
                                         subscriber_sca_entity_id,
                                         composite_sca_entity_id,
                                         subscriber_sca_entity_id,
                                         'REJ_MSG',
                                         composite_sca_entity_id,
                                         '{http://schemas.oracle.com/bpel/extension}remoteFault',
                                         'SYSTEM',
                                         edn_message.ERROR_MSG,
                                         edn_message.SHORT_ERROR_MSG,
                                         1280,
                                         composite_partition_id,
                                         composite_partition_id,
                                         composite_id,
                                         edn_message.LOCAL_NAME,
                                         edn_message.NAMESPACE,
                                         edn_message.CREATION_TIME,
                                         edn_message.CREATION_TIME
                                    );
                       log_debug(' Inserted into SCA_COMMON_FAULT', log_file);
                       -- insert_error_log(thread, 'upgrade_edn_message',0, -30022, 'Inserted into SCA_COMMON_FAULT fauld_id : ' || fault_id || ',flow_id :' || 
                       --                          iflow_id ||' for ' ||  edn_message.MSG_ID);
            EXCEPTION
                 WHEN OTHERS THEN
                     log_error('Unable to insert into SCA_COMMON_FAULT for ' || edn_message.MSG_ID || ' due to ' || SQLERRM ||  ', the script will continue', log_file);
                    insert_error_log(thread, 'upgrade_edn_message',2, -30023, 'Unable to insert into SCA_COMMON_FAULT fault_id : ' || fault_id || ', flow_id : ' ||
                                              iflow_id ||' for ' ||  edn_message.MSG_ID || ' #' || SQLCODE || '-' || SQLERRM);
                     return 2;
            END;
            BEGIN
                  log_debug('Calling insert into SCA_REJECTED_MESSAGE', log_file); 
                  insert_rejected_message(srm_id, iflow_id, edn_message.MSG_ID , edn_message.CREATION_TIME,to_clob(edn_message.ERROR_MSG), 
                   to_clob(edn_message.SHORT_ERROR_MSG),0, edn_message.NAMESPACE, edn_message.LOCAL_NAME,upgraded_target, 1280, 'utf-8',
                   clob2Blob(edn_message.PAYLOAD),512, utl_raw.cast_to_raw(''),'subscribe', composite_partition_id); 
                   log_debug('Inserted into SCA_REJECTED_MESSAGE :' ||
                                          ' srm_id :' || srm_id ||
                                         ' flow id :' || iflow_id , log_file);
            EXCEPTION
                       WHEN OTHERS THEN
                           log_error('Unable to insert into SCA_REJECTED_MESSAGE for ' || edn_message.MSG_ID || ' due to ' || SQLERRM, log_file);
                           insert_error_log(thread, 'upgrade_edn_message',2, -30024, 'Unable to insert into SCA_REJECTED_MESSAGE for ' ||  edn_message.MSG_ID || ' # ' || SQLCODE || '-' || SQLERRM);
                           return 2 ;
            END;
            BEGIN
                  update_edn_error_store(edn_message.MSG_ID, iflow_id);
            EXCEPTION
                  WHEN OTHERS THEN
                      log_error('Unable to update EDN_EVENT_ERROR_STORE for ' || edn_message.MSG_ID || ' due to ' || SQLERRM || ', the script will continue', log_file);
                      insert_error_log(thread, 'upgrade_rejected_message',2, -30025, 'Unable to update EDN_EVENT_ERROR_STORE for ' ||  edn_message.MSG_ID || ', flow_id : ' ||
                                                iflow_id || ' # ' || SQLCODE || '-' || SQLERRM);
                      return 2;
            END;
            return 0;
    END;
    
    PROCEDURE insert_rejected_message
    (
        srm_id IN NUMBER, 
        iflow_id  IN NUMBER, 
        message_origin_reference VARCHAR2,
        created_time IN timestamp,
        error_message CLOB,
        stack_trace IN CLOB,
        retry_count IN NUMBER,
        namespace IN VARCHAR2, 
        local_name IN VARCHAR2,
        srm_edn_target IN VARCHAR2,
        srm_state IN NUMBER,
        msg_encoding IN VARCHAR2,
        msg_payload IN BLOB,
        payload_type IN NUMBER,
        tracking_context_properties IN BLOB,
        resubmit_op IN VARCHAR2,
        partition_id IN NUMBER
    )
    IS
    BEGIN
       INSERT INTO SCA_REJECTED_MESSAGE
                                   (
                                     SRM_ID,
                                     FLOW_ID,
                                     MESSAGE_ORIGIN_REFERENCE,
                                     CREATED_TIME,
                                     ERROR_MESSAGE,
                                     STACK_TRACE,
                                     RETRY_COUNT,
                                     NAMESPACE,
                                     LOCAL_NAME,
                                     SRM_EDN_TARGET,
                                     SRM_STATE,
                                     MSG_ENCODING,
                                     MSG_PAYLOAD,
                                     PAYLOAD_TYPE,
                                     TRACKING_CONTEXT_PROPERTIES,
				     ERROR_CATEGORY,
                                     RESUBMIT_OP,
                                     SCA_PARTITION_ID,
                                     UPGRADED
                                    )
                                    VALUES (
                                     srm_id,
                                     iflow_id,
                                     NULL,
                                     created_time,
                                     error_message,
                                     stack_trace,
                                     retry_count,
                                     namespace,
                                     local_name,
                                     srm_edn_target,
                                     srm_state,
                                     msg_encoding,
                                     msg_payload,
                                     payload_type,
                                     tracking_context_properties,
				     's',
                                     resubmit_op,
                                     partition_id,
                                     1
                                    );
    END;

    PROCEDURE update_edn_error_store
    (
        in_id      IN RAW,
        in_flow_id IN NUMBER
    )
    IS 
    BEGIN
        UPDATE EDN_EVENT_ERROR_STORE SET FLOW_ID = in_flow_id where MSG_ID = in_id;
    END;

    PROCEDURE mark_edn_message_error
    (
        in_id      IN RAW,
        in_flow    IN NUMBER,
        log_file IN utl_file.file_type      
    )
    IS 
    BEGIN
        log_error('Error occured while upgrading edn message ' || in_id || ', flow ' || in_flow, log_file);
        update_edn_error_store(in_id, in_flow);
    EXCEPTION
        WHEN OTHERS THEN 
            log_error('Unable to update faulted edn message ' || in_id || ' ignoring this error', log_file);
    END;

END edn_migrate;
/
show errors;
